/****************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2021-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
** Copyright (C) 2017 - 2021 Jolla Ltd.
**
****************************************************************************/

import QtQuick 2.5
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Sailfish.Pickers 1.0
import Nemo.FileManager 1.0
import Sailfish.FileManager 1.0 as FileManager
import Aurora.Controls 1.0

PickerDialog {
    id: page

    property alias path: fileModel.path
    property alias includeDirectories: fileModel.includeDirectories
    property alias includeHiddenFiles: fileModel.includeHiddenFiles
    property alias includeSystemFiles: fileModel.includeSystemFiles
    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias directorySort: fileModel.directorySort
    property alias nameFilters: fileModel.nameFilters
    property alias caseSensitivity: fileModel.caseSensitivity
    property alias showParentDirectory: gridView.showParentDirectory
    property alias gridLayout: gridView.gridLayout

    function updateSelectedFiles(selectionFlag) {
        for (var i = 0; i < fileModel.count; i++) {
            if (!fileModel.data(fileModel.index(i, 0), FileModel.IsDirRole)) {
                var path = fileModel.data(fileModel.index(i, 0), FileModel.AbsolutePathRole)
                var fileItem = {
                    'url': fileModel.data(fileModel.index(i, 0), FileModel.UrlRole),
                    'title': fileModel.data(fileModel.index(i, 0), FileModel.FileNameRole),
                    'lastModified': fileModel.data(fileModel.index(i, 0), FileModel.LastModifiedRole),
                    'filePath': fileModel.data(fileModel.index(i, 0), FileModel.AbsolutePathRole),
                    'fileName': fileModel.data(fileModel.index(i, 0), FileModel.FileNameRole),
                    'fileSize': fileModel.data(fileModel.index(i, 0), FileModel.SizeRole),
                    'mimeType': fileModel.data(fileModel.index(i, 0), FileModel.MimeTypeRole),
                    'selected': fileModel.data(fileModel.index(i, 0), FileModel.IsSelectedRole),
                    'contentType': ContentType.File,
                    "fileModel": fileModel
                }

                if (_selectedModel) {
                    if (selectionFlag) {
                        if (!_selectedModel.selected(path)) {
                            _selectedModel.append(fileItem)
                        }
                    } else {
                        if (_selectedModel.selected(path)) {
                            _selectedModel.remove(_selectedModel._indexInModel(fileItem))
                        }
                    }
                }
            }
        }
    }

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: _background ? gridView : __silica_applicationwindow_instance.contentItem
        targetPage: page
    }

    FileGridView {
        id: gridView

        property bool showParentDirectory
        readonly property int fileCount: fileModel.count - fileModel.countFolders

        function selectFileByModel() {
            for (var i = 0; i < fileModel.count; ++i) {
                var absolutePath = fileModel.fileNameAt(i)
                var selectedInSelectedModel = _selectedModel.selected(absolutePath)
                if (selectedInSelectedModel) {
                    fileModel.toggleSelectedFile(i)
                }
            }
        }

        anchors.fill: parent
        anchors.bottomMargin: toolBar.height + SafeZoneRect.insets.bottom
        cellWidthForSmallScreen: parent.width
        currentIndex: -1
        selectionEnabled: true

        header: PickerDialogHeader {
            showBack: true
            selectedCount: page._selectedCount
            _glassOnly: _background
            fileSystemAccepted: true
        }

        model: FileModel {
            id: fileModel

            onPopulatedChanged: if (populated) gridView.selectFileByModel()

            path: StandardPaths.home
            active: page.status === PageStatus.Active
            includeDirectories: true
            sortBy: FileModel.SortByName
            directorySort: FileModel.SortDirectoriesBeforeFiles
            includeParentDirectory: gridView.showParentDirectory
        }

        delegate: FileManager.FileGridItem {
            id: fileDelegate

            function updateSelectedFile() {
                var fileItem = {
                    'url': model.url,
                    'title': model.fileName,
                    'lastModified': model.modified,
                    'filePath': model.absolutePath,
                    'fileName': model.fileName,
                    'fileSize': model.size,
                    'mimeType': model.mimeType,
                    'selected': model.isSelected,
                    'contentType': ContentType.File,
                    "fileModel": fileModel
                }

                if (_selectedModel) {
                    _selectedModel.update(fileItem)
                }
            }

            onClicked: {
                if (model.isDir) {
                    fileModel.clearSelectedFiles()
                    _selectedModel.clear()
                    var obj = pageStack.animatorPush('DirectoryDialog.qml', {
                                                         path: model.absolutePath,
                                                         acceptText: page.acceptText,
                                                         includeHiddenFiles: fileModel.includeHiddenFiles,
                                                         sortBy: fileModel.sortBy,
                                                         sortOrder: fileModel.sortOrder,
                                                         directorySort: fileModel.directorySort,
                                                         nameFilters: fileModel.nameFilters,
                                                         caseSensitivity: fileModel.caseSensitivity,
                                                         _selectedModel: page._selectedModel,
                                                         acceptDestination: page.acceptDestination,
                                                         acceptDestinationAction: page.acceptDestinationAction,
                                                         _animationDuration: page._animationDuration,
                                                         _background: page._background,
                                                         _clearOnBackstep: false,
                                                         gridLayout: gridLayout,
                                                         allowedOrientations: page.allowedOrientations,
                                                     })
                    obj.pageCompleted.connect(function(nextPage) {
                        nextPage.accepted.connect(function() {
                            page._dialogDone(DialogResult.Accepted)
                        })
                    })
                } else {
                    fileModel.toggleSelectedFile(index)
                    updateSelectedFile()
                }
            }

            onPressAndHold: if (!model.isDir) popupMenu.open()

            fileName: model.absolutePath
            down: model.isSelected || (_selectedModel && _selectedModel.selected(model.absolutePath)) || pressed

            PreviewPopupMenu {
                id: popupMenu

                fileName: model.url
            }
        }

        ViewPlaceholder {
            text: {
                if (page.nameFilters.length) {
                    //: Empty state text if no files match the filter
                    //% "No files match filter"
                    return qsTrId("components_pickers-ph-no_matching_files")
                }
                //: Empty state text if the directory contains no content
                //% "Empty folder"
                return qsTrId("components_pickers-ph-empty_folder")
            }
            enabled: fileModel.populated && fileModel.count === 0 && !page.showParentDirectory
        }

        VerticalScrollDecorator {}
    }

    TextMetrics {
        id: appBarTextMetrics

        //% "Clear selection"
        text: qsTrId("pickers-me-clear_selection")
                //% "Select all"
              + qsTrId("pickers-me-select_all")
                //% "More actions"
              + qsTrId("components_pickers-la-more-actions")
        font {
            pixelSize: Theme.fontSizeSmall
            family: Theme.fontFamily
        }
    }

    AppBar {
        id: toolBar

        property bool longVersion: toolBar.width > (2 * Theme.paddingLarge
                                                    + Theme.dp(128) + Theme.dp(72 * 3)
                                                    + 1.5 * Theme.horizontalPageMargin
                                                    + appBarTextMetrics.width)

        _bottomPosition: true
        background: true
        divider: false

        AppBarSpacer {
            visible: toolBar.longVersion
        }

        AppBarButton {
            onClicked: {
                fileModel.clearSelectedFiles()
                page.updateSelectedFiles(false)
            }
            //% "Clear selection"
            text: toolBar.longVersion ? qsTrId("pickers-me-clear_selection") : ""
            icon.source: "image://theme/icon-splus-multiclean"
        }

        AppBarSpacer {
            visible: toolBar.longVersion
            fixedWidth: Theme.dp(64)
        }

        AppBarButton {
            onClicked: {
                fileModel.selectFilesWithoutDirectories()
                page.updateSelectedFiles(true)
            }

            //% "Select all"
            text: toolBar.longVersion ? qsTrId("pickers-me-select_all") : ""
            icon.source: "image://theme/icon-splus-multiselect"
        }

        AppBarSpacer {
            fixedWidth: toolBar.longVersion ? Theme.dp(64) : 0
        }

        AppBarButton {
            onClicked: morePopupMenu.open()

            //% "More actions"
            text: toolBar.longVersion ? qsTrId("components_pickers-la-more-actions") : ""
            icon.source: "image://theme/icon-splus-more"

            PopupMenu {
                id: morePopupMenu

                PopupSubMenuItem {
                    //% "Sort by"
                    text: qsTrId("components_pickers-he-sort-by")
                    hint: {
                        switch (fileModel.sortBy) {
                        case FileModel.SortByName:
                            return qsTrId("components_pickers-la-sort_name")
                        case FileModel.SortBySize:
                            return qsTrId("components_pickers-la-sort_size")
                        case FileModel.SortByModified:
                            return qsTrId("components_pickers-la-sort_date")
                        case FileModel.SortByExtension:
                            return qsTrId("components_pickers-la-sort_extension")
                        }
                        return ""
                    }

                    icon.source: "image://theme/icon-splus-sort-by"

                    headerBackground: PopupMenuHeader.SolidBackground

                    SortPopupMenuItem {
                        //% "Name"
                        text: qsTrId("components_pickers-la-sort_name")
                        sortBy: FileModel.SortByName
                        fileModel: fileModel
                    }

                    SortPopupMenuItem {
                        //% "Size"
                        text: qsTrId("components_pickers-la-sort_size")
                        sortBy: FileModel.SortBySize
                        fileModel: fileModel
                    }

                    SortPopupMenuItem {
                        //% "Date modified"
                        text: qsTrId("components_pickers-la-sort_date")
                        sortBy: FileModel.SortByModified
                        fileModel: fileModel
                    }

                    SortPopupMenuItem {
                        //% "Extension"
                        text: qsTrId("components_pickers-la-sort_extension")
                        sortBy: FileModel.SortByExtension
                        fileModel: fileModel
                    }
                }

                PopupMenuItem {
                    onClicked: includeHiddenFiles= !includeHiddenFiles

                    text: includeHiddenFiles
                            //% "Hide hidden files"
                          ? qsTrId("components_pickers-la-hide_hidden_files")
                            //% "Show hidden files"
                          : qsTrId("components_pickers-la-show_hidden_files")
                    icon.source: "image://theme/icon-splus-file-%1".arg(includeHiddenFiles ? "hide" : "show")
                }

                PopupMenuItem {
                    onClicked: gridLayout = !gridLayout

                    text: gridLayout
                            //% "Show list"
                          ? qsTrId("components_pickers-la-show_grid")
                            //% "Show grid"
                          : qsTrId("components_pickers-la-show_list")
                    icon.source: "image://theme/icon-splus-%1".arg(gridLayout ? "list" : "cage")
                }
            }
        }

        AppBarSpacer {
            visible: toolBar.longVersion
        }
    }
}
