/**
 * SPDX-FileCopyrightText: Copyright 2022-2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Sailfish.Share 1.0
import org.nemomobile.configuration 1.0
import org.nemomobile.systemsettings 1.0
import Aurora.Controls 1.0

Page {
    id: searchPage

    property int accountId
    property string searchString: searchField.text.toLowerCase().trim()
    property bool openingTopPage
    property bool resetSearch
    property string path
    property string initialPath
    property var caseSensitivity
    property bool openSymLink: true
    property var searchModel: SearchProxyModel {
        pattern: searchString
        sourceModel: fileModel
    }
    property ExtractorView extractor
    property int __search_page
    property bool multiselection
    property AppBarSearchField searchField: appBar.searchField

    readonly property var standardPaths: [
        StandardPaths.documents,
        StandardPaths.download,
        StandardPaths.pictures,
        StandardPaths.music,
        StandardPaths.videos,
        StandardPaths.home + "/Playlists",
        StandardPaths.home + "/Templates",
        StandardPaths.home + "/Public",
        StandardPaths.home + "/Desktop"
    ]

    function _closeVirtualKeyboard() {
        Qt.inputMethod.hide()
        searchField.focus = false
    }

    function openDirectory(dirPath) {
        FileManager.openDirectory({
                                      path: dirPath,
                                      initialPath: searchPage.path,
                                      caseSensitivity: searchPage.caseSensitivity,
                                      extractor: searchPage.extractor
                                  })
    }

    ConfigurationValue {
        id: hiddenItemsSetting

        key: "/apps/sailfish-filemanager/show_hidden_items"
        defaultValue: false

        onValueChanged: {
            TrackerEngine.showHidden = hiddenItemsSetting.value
        }
    }

    onStatusChanged: {
        if (status === PageStatus.Active && !searchPage.multiselection)
            searchField.forceActiveFocus()

        if (status == PageStatus.Activating) {
            TrackerEngine.currentPath = searchPage.path.length > 0 ? searchPage.path : ""
            TrackerEngine.showHidden = hiddenItemsSetting.value
            extractor.isOpen = false
            extractor.archiveFile = ""
            extractor.passwordNeeded = false
        }
    }

    onSearchStringChanged: {
        if(searchString.length > 0) {
            busyIndicator.running = gridView.count === 0
            waitTimer.restart()
        } else {
            searchField.forceActiveFocus()
        }
    }

    Component.onCompleted: {
        SortingSettings.setSortType(fileModel.sortBy, fileModel.sortOrder, fileModel.directorySort)
        extractor.rejectedPageName = '__search_page'
        extractor.extractHereAvailable = false
    }

    TrackerModel {
        id: fileModel

        path: searchPage.path
        trackerData: TrackerEngine.data
        directorySort: SortingSettings.directorySort
        sortBy: SortingSettings.sortBy
        sortOrder: SortingSettings.sortBy === FileModel.SortBySize
                   ? SortingSettings.sortOrder == Qt.AscendingOrder ? Qt.DescendingOrder : Qt.AscendingOrder
                   : SortingSettings.sortOrder
    }

    FileGridHeader {
        id: appBar

        parentGridView: gridView
        errorType: fileModel.errorType
        path: fileModel.path
        countFiles: searchModel.count
        popUpMenuSort: sortPopUpMenu
        isSelectionMode: gridView.selectionMode
        isPasteMode: gridView.pasteMode && !isArchiveMode
        isArchiveMode: gridView.archiveMode
        isExtractMode: gridView.extractMode
        isSearchMode: gridView.searchPage
        z: searchPage.z + 1

        onSelectModeClicked: {
            pageStack.animatorPush(Qt.resolvedUrl("SearchPage.qml"),
                                   {
                                       path: searchPage.path,
                                       initialPath: searchPage.initialPath,
                                       caseSensitivity: searchPage.caseSensitivity,
                                       extractor: searchPage.extractor,
                                       searchModel: searchPage.searchModel,
                                       searchString: searchPage.searchString,
                                       multiselection: true
                                   },
                                   PageStackAction.Animated)
        }

        searchField {
            visible: !searchPage.multiselection

            enabled: !gridView.isSelectOneItem && !gridView.selectionMode
                     && !gridView.pasteMode && !gridView.extractMode

            //% "Search"
            placeholderText: qsTrId("filemanager-la-search_files")
        }

        onUpdateButtonVisibilityFinished: FileManager.updatePopUpContent()
    }

    PopupMenuSort {
        id: sortPopUpMenu

        onOpenedChanged: searchPage.showNavigationIndicator = !opened
        onSortClicked: {
            if (sortBy !== fileModel.sortBy || sortOrder !== fileModel.sortOrder) {
                SortingSettings.setSortType(sortBy, sortOrder, directorySort)
                TrackerEngine.setSortOptions(sortBy, sortOrder)
            }
        }
    }

    FileGridView {
        id: gridView

        onMovementStarted: searchPage._closeVirtualKeyboard()

        anchors {
            fill: parent
            topMargin: appBar.height

        }
        cellWidthForSmallScreen: gridView.width
        model: searchModel
        gridLayout: FileManager.gridLayoutSetting.value
        searchPage: true
        extractor: searchPage.extractor
        standardPaths: searchPage.standardPaths
        selectionMode: searchPage.multiselection

        header: Item {
            width: headerContainer.width
            height: headerContainer.height
        }

        ShareAction {
            id: shareAction
        }

        delegate: FileGridItem {
            id: delegateItem

            fileName: model.absolutePath
            trackerMode: true
            mimeTypeCallback: function() { return TrackerEngine.mimeTypeAt(model.absolutePath) }
            searchText: searchPage.searchString
            showStorageIcon: externalStorageModel.count > 0
            visible: !busyIndicator.running
            down: gridView.selectionMode
                  ? searchModel.selectedFiles().indexOf(model.absolutePath) !== -1 ||
                    FileEngine.clipboardFiles.indexOf(model.absolutePath) !== -1 || pressed
                    || (isArchive && extractor.archiveFile === fileName)
                  : pressed || (searchModel.selectedFiles().indexOf(model.absolutePath) !== -1
                                && gridView.isSelectOneItem)
            enabled: {
                if (!gridView.pasteMode && !gridView.extractMode && !gridView.isSelectOneItem) {
                    return true
                } else {
                    return model.isDir
                }
            }
            opacity: enabled ? 1.0 : Theme.opacityLow

            onClicked: {
                if (gridView.selectionMode) {
                    if (down)
                        info.file = ""
                    else
                        info.file = model.absolutePath

                    searchModel.toggleSelectedFile(model.index)

                } else if (model.isDir) {
                    var dirPath = fileModel.appendPath(model.absolutePath)
                    if (!FileEngine.isSymLink(dirPath)) {
                        openDirectory(dirPath)
                    } else {
                        if (openSymLink) {
                            openDirectory(dirPath)
                        } else {
                            //% "Following a symbolic link is prohibited"
                            errorNotification.show(qsTrId("filemanager-la-open_symlink_failed"))
                        }
                    }
                } else {
                    if (!FileEngine.isArchive(model.absolutePath)) {
                        if (!FileManager.openGalleryPreviewPage(model.absolutePath,
                                                                delegateItem.mimeType,
                                                                searchPage.extractor))
                            Qt.openUrlExternally(FileManager.pathToUrl(model.absolutePath))
                    } else {
                        FileManager.FileManager.openArchive(model.absolutePath,
                                                            "/",
                                                            model.path,
                                                            searchPage.extractor,
                                                            true,
                                                            archivePageComponent,
                                                            '__search_page')
                    }
                }
            }

            onPressAndHold: {
                if (!gridView.selectionMode && !gridView.pasteMode && !gridView.extractMode && !gridView.isSelectOneItem) {
                    gridView.currentIndex = model.index
                    gridView.isSelectOneItem = true
                    searchModel.toggleSelectedFile(model.index)
                }
            }

            Component {
                id: archivePageComponent

                ArchivePage {}
            }
        }

        FileInfo {
            id: info
        }

        ViewPlaceholder {
            onEnabledChanged: if (enabled) busyIndicator.running = false
            enabled: gridView.count === 0 && searchField.text.length > 0
            //% "No files found"
            text: qsTrId("filemanager-la-no_files_found")
        }

        Column {
            id: headerContainer

            width: searchPage.width
            parent: gridView.contentItem
            anchors.top: gridView.headerItem ? gridView.headerItem.top : gridView.top

            Item {
                width: parent.width
                height: Theme.paddingMedium
            }

            SectionHeader {
                objectName: "SortedBySectionHeader"

                x: Theme.paddingLarge
                text: qsTrId("filemanager-la-sort").arg(SortingSettings.selectedSortType)
                horizontalAlignment: Text.AlignLeft
                opacity: searchModel.count > 0 ? 1 : 0

                Behavior on opacity { FadeAnimation {} }
            }
        }

        VerticalScrollDecorator {}
    }

    PageBusyIndicator {
        id: busyIndicator

        z: 1
        FadeAnimator {
            target: busyIndicator
            duration: 1000
            easing.type: Easing.InExpo
            to: 1.0
        }
    }

    Timer {
        id: waitTimer

        interval: 150
        onTriggered: {
            busyIndicator.running = false
        }
    }

    PartitionModel {
        id: externalStorageModel

        storageTypes: PartitionModel.External | PartitionModel.ExcludeParents
    }

    Connections {
        target: FileManager
        onSelectedFilesCleared: searchModel.clearSelectedFiles()
    }
}
