/****************************************************************************
**
** Copyright (с) 2021 - 2022 Open Mobile Platform LLC.
**
** License: Proprietary.
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0

Dialog {
    id: dialog
    objectName: "NewFolderDialog"

    property string path
    readonly property string _newPath: folderName.text !== ""
            ? path + "/" + folderName.text
            : ""

    readonly property var _regExp: new RegExp("[\/*\?<>\|]+")
    readonly property bool _hasInvalidCharacters: _regExp.test(folderName.text)
    readonly property bool _exists: _newPath !== "" && FileEngine.exists(_newPath)
    readonly property bool _hasFirstSpace: _newPath !== "" && /^\s+/.test(folderName.text)
    readonly property bool _hasFirstDot: _newPath !== "" && folderName.text.charAt(0) === "." && folderName.length === 1
    readonly property bool _tooLongName: encodeURI(folderName.text).split(/%..|./).length - 1 > 255

    SilicaFlickable {
        id: flickable
        objectName: "NewFolderDialogSilicaFlickable"

        anchors.fill: parent
        contentHeight: column.height

        Column {
            id: column
            width: parent.width

            DialogHeader {
                objectName: "CreateDialogHeader"

                //% "Create"
                acceptText: qsTrId("filemanager-he-create")
            }
            TextField {
                id: folderName
                objectName: "FolderNameTextField"

                width: parent.width
                //% "New folder"
                placeholderText: qsTrId("filemanager-la-new_folder")
                label: {
                    if (dialog._hasInvalidCharacters || dialog._hasFirstDot) {
                        //% "Invalid folder name"
                        return qsTrId("filemanager-te-invalid_foldername")
                    } else if (dialog._exists) {
                        //% "A folder with the same name exists"
                        return qsTrId("filemanager-te-foldername_exists")
                    } else if (dialog._hasFirstSpace) {
                        //% "Folder name can`t start with spaces"
                        return qsTrId("filemanager-te-foldername_can`t_start_with_spaces")
                    } else if (dialog._tooLongName) {
                        //% "Folder name is too long"
                        return qsTrId("filemanager-te-foldername_too_long")
                    } else {
                        return qsTrId("filemanager-la-new_folder")
                    }
                }

                errorHighlight: dialog._hasInvalidCharacters || dialog._exists || dialog._hasFirstSpace || dialog._hasFirstDot || dialog._tooLongName
                focus: true

                EnterKey.enabled: text.length > 0
                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: dialog.accept()
            }
        }
        VerticalScrollDecorator {}
    }

    canAccept: _newPath !== "" && !_exists && !_hasFirstSpace && !_hasFirstDot && !_hasInvalidCharacters && !_tooLongName
    onAccepted: {
        if (!FileEngine.mkdir(path, folderName.text, true)) {
            //% "Cannot create folder %1"
            errorNotification.show(qsTrId("filemanager-la-cannot_create_folder").arg(folderName.text))
        }
    }
}
