/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: BSD 3-Clause
**
****************************************************************************************/


import QtQuick 2.4
import Sailfish.Silica 1.0

SilicaItem {
    id: root

    property real progress
    property bool compact

    width: Theme.dp(48)
    height: Theme.dp(72)
    scale: 1 + progress * 0.25

    Rectangle {
        id: highlight

        anchors.fill: parent

        color: Qt.tint(Qt.tint(palette.highlightColor,
                               palette.colorScheme === Theme.DarkOnLight ? "#ccffffff" : "#cc000000"), // 80%
                       Theme.rgba(palette.highlightColor, Theme.opacityFaint))
        opacity: progress
        radius: Theme.dp(8)
    }

    ShaderEffect {
        id: shader

        property real frame: {
            if (!compact) {
                return 0
            }

            return (1 - progress) * (frameCount - 1)
        }

        property int frameWidth: 18

        property real frameStart: parseInt(frame) / frameCount
        property real frameEnd: (parseInt(frame) + 1) / frameCount
        property real morphCoef: frame - parseInt(frame)
        property int frameCount: parseInt(sprite.width / frameWidth)
        property real frameCountCoef: 1.0 / frameCount

        property variant source: sprite
        property color color: root.palette.primaryColor

        x: {
            const center = (width + parent.width) * 0.5 - progress * Theme.dp(1)
            return center - (center * (frame / (frameCount - 1))) - width
        }
        anchors.verticalCenter: parent.verticalCenter
        anchors.verticalCenterOffset: Theme.dp(1)
        width: frameWidth
        height: sprite.height
        scale: Theme.pixelRatio

        fragmentShader: "
                varying highp vec2 qt_TexCoord0;
                uniform lowp float qt_Opacity;
                uniform highp vec4 qt_SubRect_source;
                uniform highp float frameStart;
                uniform highp float frameEnd;
                uniform lowp float morphCoef;
                uniform lowp vec4 color;
                uniform lowp float frameCountCoef;
                uniform lowp sampler2D source;

                void main() {
                    highp vec2 coord = vec2(qt_TexCoord0.x * frameCountCoef, qt_TexCoord0.y);

                    highp vec4 col = texture2D(source, qt_SubRect_source.xy + qt_SubRect_source.zw * (coord + vec2(frameStart, 0.0)));
                    highp vec4 col2 = texture2D(source, qt_SubRect_source.xy + qt_SubRect_source.zw * (coord + vec2(frameEnd, 0.0)));

                    highp vec4 outColor = mix(col, col2, morphCoef);

                    gl_FragColor = outColor * color * qt_Opacity;
                }
            "
    }

    Image {
        id: sprite

        visible: false
        source: "chevron-sprite.png"
    }
}
