/****************************************************************************
**
** Copyright (c) 2021 - 2023 Open Mobile Platform LLC.
** Copyright (C) 2016 - 2021 Jolla Ltd.
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Nemo.FileManager 1.0
import Sailfish.Pickers 1.0

PickerPage {
    id: page

    property alias path: fileModel.path
    property alias includeDirectories: fileModel.includeDirectories
    property alias includeHiddenFiles: fileModel.includeHiddenFiles
    property alias includeSystemFiles: fileModel.includeSystemFiles
    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias directorySort: fileModel.directorySort
    property alias nameFilters: fileModel.nameFilters
    property alias caseSensitivity: fileModel.caseSensitivity
    property alias showParentDirectory: gridView.showParentDirectory

    signal filePicked(var path)

    function contentTypeForFile(file) {
        if (file.isVideo) {
            return ContentType.Video
        } else if (file.isPicture) {
            return ContentType.Image
        } else if (!file.isDir) {
            var imageName = Theme.iconForMimeType(file.mimeType)

            if (imageName.indexOf("audio") !== -1) {
                return ContentType.Audio
            } else if (imageName.indexOf("note") !== -1
                       || imageName.indexOf("pdf") !== -1
                       || imageName.indexOf("spreadsheet") !== -1
                       || imageName.indexOf("presentation") !== -1
                       || imageName.indexOf("formatted") !== -1
                       || imageName.indexOf("document") !== -1) {
                return ContentType.Document
            }
        }

        return ContentType.File
    }

    title: fileModel.directoryName

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: _background ? gridView : __silica_applicationwindow_instance.contentItem
        targetPage: page
    }

    FileGridView {
        id: gridView

        property bool showParentDirectory

        anchors.fill: parent
        cellWidthForSmallScreen: parent.width
        currentIndex: -1
        selectionEnabled: false

        header: PageHeader {
            title: page.title
        }

        model: FileModel {
            id: fileModel
            active: page.status === PageStatus.Active
            path: StandardPaths.home
            includeDirectories: true
            sortBy: FileModel.SortByName
            directorySort: FileModel.SortDirectoriesBeforeFiles

            includeParentDirectory: gridView.showParentDirectory
        }

        delegate: GridItem {
            onClicked: {
                if (model.isDir) {
                    var obj = pageStack.animatorPush('DirectoryPage.qml', {
                                                         path: model.absolutePath,
                                                         includeHiddenFiles: fileModel.includeHiddenFiles,
                                                         sortBy: fileModel.sortBy,
                                                         sortOrder: fileModel.sortOrder,
                                                         directorySort: fileModel.directorySort,
                                                         nameFilters: fileModel.nameFilters,
                                                         caseSensitivity: fileModel.caseSensitivity,
                                                         allowedOrientations: page.allowedOrientations
                                                     })
                    obj.pageCompleted.connect(function(nextPage) {
                        nextPage.filePicked.connect(page.filePicked)
                    })
                } else {
                    page.filePicked({
                                        'url': model.url,
                                        'title': model.fileName,
                                        'lastModified': model.modified,
                                        'filePath': model.absolutePath,
                                        'fileName': model.fileName,
                                        'fileSize': model.size,
                                        'mimeType': model.mimeType,
                                        'selected': true,
                                        'contentType': page.contentTypeForFile(fileItem)
                                    })
                }
            }

            contentWidth: fileItem.width
            contentHeight: fileItem.height
            _backgroundColor: "transparent"
            propagateComposedEvents: true
            menu: menuComponent

            FileGridItem {
                id: fileItem

                width: gridView.cellWidth
                chunkWidth: gridView.chunkWidth
                fileName: model.absolutePath
                size: model.size
                modified: model.modified
                selectionEnabled: false
            }

            Connections {
                target: gridView
                onWidthChanged: if (gridView.largeScreen && _menuItem && _menuItem.parent) _calculateMenuOffset()
            }

            Component {
                id: menuComponent

                ContextMenu {
                    MenuItem {
                        text: qsTrId("components_pickers-he-file_preview")
                        onClicked: Qt.openUrlExternally(model.absolutePath)
                    }
                }
            }
        }

        ViewPlaceholder {
            text: {
                if (page.nameFilters.length) {
                    //: Empty state text if no files match the filter
                    //% "No files match filter"
                    return qsTrId("components_pickers-ph-no_matching_files")
                }
                //: Empty state text if the directory contains no content
                //% "Empty folder"
                return qsTrId("components_pickers-ph-empty_folder")
            }
            enabled: fileModel.populated && fileModel.count === 0 && !page.showParentDirectory
        }

        PickerPullDownMenu {
            sourceModel: fileModel
            currentPage: page
            currentView: gridView
        }

        VerticalScrollDecorator {}
    }
}
