/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD 3-Clause
**
** Copyright (C) 2013 Jolla Ltd.
** All rights reserved.
**
** This file is part of Sailfish Silica UI component package.
**
** You may use this file under the terms of BSD license as follows:
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are met:
**     * Redistributions of source code must retain the above copyright
**       notice, this list of conditions and the following disclaimer.
**     * Redistributions in binary form must reproduce the above copyright
**       notice, this list of conditions and the following disclaimer in the
**       documentation and/or other materials provided with the distribution.
**     * Neither the name of the Jolla Ltd nor the
**       names of its contributors may be used to endorse or promote products
**       derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
** DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
** ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
****************************************************************************************/

import QtQuick 2.2
import Sailfish.Silica 1.0

SilicaItem {
    id: root

    property bool running
    property int size: BusyIndicatorSize.Medium
    property alias color: busyIndicator.color
    property bool _forceAnimation

    implicitWidth: busyIndicator.width
    implicitHeight: busyIndicator.height
    opacity: running ? 1.0 : 0.0
    Behavior on opacity { FadeAnimation {}}

    ShaderEffect {
        id: busyIndicator

        readonly property real antialias: 4 / width
        property color color: palette.highlightColor
        property real time

        anchors.centerIn: parent

        width: switch (root.size) {
            case BusyIndicatorSize.ExtraSmall:
                return Theme.iconSizeExtraSmall
            case BusyIndicatorSize.Small:
                return Theme.iconSizeSmall
            case BusyIndicatorSize.Medium:
                return Theme.iconSizeMedium
            default: // BusyIndicatorSize.Large
                return Theme.iconSizeExtraLarge
        }
        height: width

        vertexShader: "
#define COUNT      10 // Should be equal to fragmentShader.COUNT and busyIndicator.count
#define PI2        1.57079632679 // pi / 2
#define TAU        6.28318530718 // pi * 2
#define START_SIZE 0.09 // default start blob size

precision lowp float;

uniform highp mat4 qt_Matrix;
attribute highp vec4 qt_Vertex;
attribute highp vec2 qt_MultiTexCoord0;

varying highp vec2 coord;
varying vec2 pointArray[COUNT];
varying float sizeArray[COUNT];
varying float opacityArray[COUNT];

uniform float time;

float easeInOutQuad(float x) {
    return x < 0.5 ? 2.0 * x * x : 1.0 - pow(-2.0 * x + 2.0, 2.0) * 0.5;
}

float easeInOutQuart(float x) {
    return x < 0.5 ? 8.0 * x * x * x * x : 1.0 - pow(-2.0 * x + 2.0, 4.0) * 0.5;
}

void main() {
    coord = 2.0 * qt_MultiTexCoord0 - 1.0;

    const float count = float(COUNT);
    const float delta = 0.14;
    const float delta2 = delta * 0.5;
    const float delta4 = delta2 * 0.5;
    const float delta8 = delta4 * 0.5;

    float dt = time * delta * count * 0.25 - delta4;
    float rotation = (360.0 / count * TAU / 180.0) * easeInOutQuart(time) - PI2;
    float scale = (0.625 + 0.375 * (0.5 - abs(0.5 - easeInOutQuad(time))));

    for (int i = 0; i < COUNT; i++) {
        float fi = float(i);
        float angle = fi / count * TAU + rotation;
        float size = delta2 - abs(delta2 - min(delta, max(0.0, dt - fi * delta8)));
        pointArray[i] = vec2(cos(angle), sin(angle)) * scale;
        sizeArray[i] = size + START_SIZE;
        opacityArray[i] = 0.4 + 0.6 * (2.0 * size / delta);
    }

    gl_Position = qt_Matrix * qt_Vertex;
}
        "

        fragmentShader: "
#define COUNT 10 // Should be equal to vertexShader.COUNT and busyIndicator.count

precision lowp float;

uniform vec4 color;
uniform float antialias;
uniform float qt_Opacity;

varying highp vec2 coord;
varying vec2 pointArray[COUNT];
varying float sizeArray[COUNT];
varying float opacityArray[COUNT];

void main() {
    float value = 0.0;
    for (int i = 0; i < COUNT; i++) {
        float blob = 1.0 - clamp((length(coord - pointArray[i]) - sizeArray[i]) / antialias, 0.0, 1.0);
        value += blob * opacityArray[i];
    }
    gl_FragColor = color * value * qt_Opacity;
}
        "

        UniformAnimator {
            target: busyIndicator
            uniform: "time"
            from: 0; to: 1
            duration: 1000
            running: (root.running || opacity > 0) && (_forceAnimation || (root.visible && Qt.application.active))
            loops: Animation.Infinite
        }
    }
}
