/*
 * Copyright (c) 2022 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */
import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Pickers 1.0
import Sailfish.FileManager 1.0 as FileManager
import Nemo.FileManager 1.0

FileManager.RenameDialog {
    id: dialog

    property bool needSelectDirectory
    property string archiveExtension
    property string currentDirectoryPath
    property int archiveType
    property ExtractorView extractorView
    property var files
    property var __directory_page: pageStack.find(function(dialog) { return dialog.hasOwnProperty('__directory_page') })

    objectName: "CreateArchiveDialog"

    onAccepted: {
        extractorView.compressFiles(files, nameField.text, directoryPath, archiveType)
    }

    //% "Create archive"
    title: qsTrId("filemanager-he-сreate_archive")
    archiveCreationMode: true
    directoryPath: currentDirectoryPath

    acceptDestination: __directory_page
    acceptDestinationAction: PageStackAction.Pop
    canAccept: _newPath !== "" && !_exists && !_hasFirstSpace && !_hasEmptyName && !_hasFirstDot && directoryPath !== ""

    ArchiveTypeModel {
        id: archiveTypeModel
    }

    Column {
        width: parent.width
        anchors.top: dialog.nameField.bottom

        ArchiveComboBoxItem {
            Component.onCompleted: {
                dialog.archiveExtension = archiveTypeModel.displayName(currentIndex)
                dialog.archiveType = archiveTypeModel.archiveType(currentIndex)
                dialog.oldPath = dialog.files.length === 1 ? dialog.files[0] + dialog.archiveExtension
                                                           : dialog.currentDirectoryPath + "/" + dialog.fileNameWithoutExt + dialog.archiveExtension
            }

            //% "Archive format"
            label: qsTrId("filemanager-la-archive_format")
            iconSource: "image://theme/icon-m-file-archive-folder"
            menu: ContextMenu {
                Repeater {
                    model: archiveTypeModel
                    MenuItem {
                        text: model.displayName
                        onClicked: {
                            var fileName = dialog.nameField.text.replace(dialog.archiveExtension, "")
                            dialog.archiveExtension = model.displayName
                            dialog.archiveType = model.type
                            fileName = fileName.concat(dialog.archiveExtension)
                            dialog.nameField.text = fileName
                        }
                    }
                }
            }
        }

        ArchiveComboBoxItem {
            //% "Archive location"
            label: qsTrId("filemanager-la-archive_location")
            iconSource: "image://theme/icon-m-file-folder"
            currentIndex: !needSelectDirectory ? 1 : 0
            value: {
                if (needSelectDirectory) {
                    //% "%1"
                    return qsTrId("filemanager-me-сreate_to").arg(dialog.directoryPath.split("/").pop())
                } else {
                    //% "Current directory"
                    return qsTrId("filemanager-me-сurrent_directory")
                }
            }

            //% "Archive will be created to %1"
            description: qsTrId("filemanager-me-will_be_saved_to_download").arg(dialog.directoryPath)
            menu: ContextMenu {
                MenuItem {
                    //% "Select a archive folder"
                    text: qsTrId("filemanager-me-select_archive_archive_folder")
                    onClicked: pageStack.animatorPush(folderPickerPage)
                }

                MenuItem {
                    text: qsTrId("filemanager-me-сurrent_directory")
                    onClicked: {
                        needSelectDirectory = false
                        directoryPath = currentDirectoryPath
                    }
                }
            }
        }
    }

    Component {
        id: folderPickerPage

        FolderPickerPage {
            showSystemFiles: false
            //% "Create to"
            dialogTitle: qsTrId("filemanager-ti-create-to")

            onSelectedPathChanged: {
                needSelectDirectory = true
                directoryPath = selectedPath
            }
        }
    }
}
