/****************************************************************************
**
** Copyright (c) 2016 – 2019 Jolla Ltd.
** Copyright (c) 2019 - 2022 Open Mobile Platform LLC
**
** License: Proprietary
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Nemo.FileManager 1.0
import Sailfish.FileManager 1.0
import org.nemomobile.configuration 1.0

Page {
    id: page

    property alias path: fileModel.path
    property alias header: gridView.header
    property string initialPath: homePath
    property string homePath // deprecated
    onHomePathChanged: if (homePath.length > 0) console.warn("DirectoryPage.homePath deprecated, use initialPath instead.")

    property string title
    property string description
    property bool showFormat
    property alias showDeleteFolder: deleteFolder.visible

    property bool mounting
    property bool showNewFolder: true
    property bool openSymLink: true

    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias caseSensitivity: fileModel.caseSensitivity
    property alias directorySort: fileModel.directorySort
    property alias errorType: fileModel.errorType

    property int __directory_page
    property string _deletingPath

    readonly property bool indexIfExists: true
    readonly property var standardPaths: [
        StandardPaths.documents,
        StandardPaths.download,
        StandardPaths.pictures,
        StandardPaths.music,
        StandardPaths.videos,
        StandardPaths.home + "/Playlists",
        StandardPaths.home + "/Templates",
        StandardPaths.home + "/Public",
        StandardPaths.home + "/Desktop"
    ]
    /*!
    \internal

    Implementation detail for file manager
    */
    property FileManagerNotification errorNotification

    signal formatClicked

    signal folderDeleted(string path)

    function refresh() {
        fileModel.refresh()
    }

    function _commitDeletion() {
        FileEngine.deleteFiles(_deletingPath, true)
    }

    function _getPasteTextId() {
        var path = FileEngine.clipboardFiles[0]

        if (FileEngine.clipboardCount > 1) {
            //: Shown when inserting a group of files, e.g. "Paste all"
            //% "Paste all"
            return qsTrId("filemanager-me-paste_all")
        } else if (path) {
            //: Shows the filename when paste, e.g. "Paste myFile.txt"
            //% "Paste %1"
            return qsTrId("filemanager-me-paste-with-arg").arg(path.split("/").pop())
        } else {
            //% "Paste"
            return qsTrId("filemanager-me-paste")
        }
    }

    backNavigation: !FileEngine.busy

    Component.onCompleted: {
        if (!errorNotification) {
            errorNotification = errorNotificationComponent.createObject(page)
        }

        FileManager.init(pageStack)
        if (!FileManager.errorNotification) {
            FileManager.errorNotification = errorNotification
        }

        FileOperationMonitor.instance()
        SortingSettings.setSortType(sortBy, sortOrder, directorySort)
    }

    FileModel {
        id: fileModel

        directorySort: SortingSettings.directorySort
        sortBy: SortingSettings.sortBy
        sortOrder: SortingSettings.sortOrder
        path: initialPath
        active: page.status === PageStatus.Active
        includeHiddenFiles: hiddenItemsSetting.value
    }

    FileGridView {
        id: gridView
        objectName: "FileGridView"

        anchors.fill: parent

        cellWidthForSmallScreen: parent.width

        model: fileModel
        opacity: {
            if (FileEngine.busy) {
                return 0.6
            } else if (page.mounting) {
                return 0.0
            } else {
                return 1.0
            }
        }

        Behavior on opacity { FadeAnimator {} }

        PullDownMenu {
            visible: fileModel.count > 0 || FileEngine.isUserDir(fileModel.path)

            MenuItem {
                objectName: "FormatMenuItem"

                //% "Format"
                text: qsTrId("filemanager-me-format")
                visible: showFormat
                onClicked: page.formatClicked()
            }

            MenuItem {
                id: deleteFolder
                objectName: "DeleteFolderMenuItem"

                //% "Delete folder"
                text: qsTrId("filemanager-me-delete_folder")
                visible: page.path !== page.initialPath && standardPaths.indexOf(fileModel.path) == -1
                onClicked: {
                    var _page = pageStack.previousPage(page)
                    if (_page && _page.hasOwnProperty("__directory_page")) {
                        pageStack.pop()
                        _page._deletingPath = page.path
                        var popup = Remorse.popupAction(_page,
                                                        Remorse.deletedText,
                                                        function() {
                                                            _page._commitDeletion()
                                                        })
                        popup.canceled.connect(function() { _page._deletingPath = "" })
                    } else {
                        //% "Deleting folder"
                        Remorse.popupAction(page, qsTrId("filemanager-la-deleting_folder"), function() {
                            FileEngine.deleteFiles(path, true)
                            folderDeleted(path)
                            pageStack.pop()
                        })
                    }
                }
            }

            MenuItem {
                objectName: "NewFolderMenuItem"

                //% "New folder"
                text: qsTrId("filemanager-me-new_folder")
                visible: page.showNewFolder && FileEngine.isUserDir(fileModel.path)
                onClicked: {
                    pageStack.animatorPush(Qt.resolvedUrl("NewFolderDialog.qml"), {
                                               path: page.path
                                           })
                }
            }

            MenuItem {
                objectName: "SortMenuItem"

                //% "Sort"
                text: qsTrId("filemanager-me-sort")
                visible: fileModel.count > 0
                onClicked: {
                    var obj = pageStack.animatorPush(Qt.resolvedUrl("SortingPage.qml"))
                    obj.pageCompleted.connect(function(dialog) {
                        dialog.selected.connect(
                                    function(sortBy, sortOrder, directorySort) {
                                        if (sortBy !== fileModel.sortBy || sortOrder !== fileModel.sortOrder) {
                                            SortingSettings.setSortType(sortBy, sortOrder, directorySort)
                                            // Wait for the changes to take effect
                                            // before popping the sorting page
                                            fileModel.sortByChanged.connect(pop)
                                            fileModel.sortOrderChanged.connect(pop)
                                        } else {
                                            pageStack.pop()
                                        }
                                    })
                    })
                }

                function pop() {
                    pageStack.pop()
                    fileModel.sortByChanged.disconnect(pop)
                    fileModel.sortOrderChanged.disconnect(pop)
                }
            }

            MenuItem {
                //% "Show"
                readonly property string showHidden: qsTrId("filemanager-me-dont_show_hidden_item")
                //% "Don`t show"
                readonly property string dontShowHidden: qsTrId("filemanager-me-show_hidden_item")

                objectName: "HiddenItemsMenuItem"

                //% "%1 hidden items"
                text: qsTrId("filemanager-me-hidden_items").arg(hiddenItemsSetting.value ? dontShowHidden : showHidden)
                visible: FileEngine.isUserDir(fileModel.path)
                onClicked: hiddenItemsSetting.value = !hiddenItemsSetting.value
            }

            MenuItem {
                objectName: "SelectItemsMenuItem"

                //% "Select items"
                text: qsTrId("filemanager-me-select_items")
                visible: fileModel.count > 0 && FileEngine.isUserDir(fileModel.path)
                onClicked: pageStack.animatorPush(Qt.resolvedUrl("MultiSelectionPage.qml"), {
                                                      selectionModel: fileModel,
                                                      extractorView: extractor,
                                                      currentDirectoryPath: page.path
                                                  })
            }

            MenuItem {
                objectName: "PasteTextIDMenuItem"

                text: _getPasteTextId()
                visible: FileEngine.clipboardCount > 0 && FileEngine.isUserDir(fileModel.path)
                onClicked: FileEngine.pasteFiles(page.path, true, indexIfExists)
            }
        }

        ConfigurationValue {
            id: hiddenItemsSetting

            key: "/apps/sailfish-filemanager/show_hidden_items"
            defaultValue: false
        }

        header: Column {
            width: parent.width
            PageHeader {
                id: pageHeader
                objectName: "PathPageHeader"

                title: path === initialPath && page.title.length > 0 ? page.title
                                                                     : page.path.split("/").pop()
                description: page.description

                BusyIndicator {
                    objectName: "BusyIndicator"

                    anchors {
                        verticalCenter: parent.verticalCenter
                        right: parent.right
                        rightMargin: Theme.paddingLarge
                    }

                    parent: pageHeader.extraContent
                    size: BusyIndicatorSize.ExtraSmall
                    running: FileEngine.mode === FileEngine.CopyMode || FileEngine.mode === FileEngine.MoveMode
                }
            }

            SectionHeader {
                objectName: "SortedBySectionHeader"

                //% "Sorted by %1"
                text: qsTrId("filemanager-la-sort").arg(SortingSettings.selectedSortType)
                opacity: fileModel.count > 0 ? 1 : 0

                Behavior on opacity { FadeAnimation {} }
            }
            // Padding to match design
            Item {
                width: parent.width
                height: Theme.paddingSmall
                opacity: gridView.largeScreen ? 1 : 0

                Behavior on opacity { FadeAnimation {} }
            }
        }

        delegate: GridItem {
            id: delegateItem
            function remove() {
                delegateItem.remorseDelete(function() { FileEngine.deleteFiles(fileModel.fileNameAt(model.index), true) })
            }

            function openDirectory(dirPath) {
                FileManager.openDirectory({
                                              path: dirPath,
                                              initialPath: page.initialPath,
                                              caseSensitivity: page.caseSensitivity,
                                          })
            }

            contentWidth: fileItem.width
            contentHeight: fileItem.height
            menu: contextMenu
            onClicked: {
                if (model.isDir) {
                    var dirPath = fileModel.appendPath(model.fileName)
                    if (!FileEngine.isSymLink(dirPath)) {
                        openDirectory(dirPath)
                    } else {
                        if (openSymLink) {
                            openDirectory(dirPath)
                        } else {
                            //% "Following a symbolic link is prohibited"
                            errorNotification.show(qsTrId("filemanager-la-open_symlink_failed"))
                        }
                    }
                } else {
                    Qt.openUrlExternally(FileManager.pathToUrl(fileModel.path + "/" + fileName))
                }
            }

            FileGridItem {
                id: fileItem
                objectName: "FileGridItem"

                width: gridView.cellWidth
                chunkWidth: gridView.chunkWidth
                fileName: model.absolutePath
            }

            Component {
                id: contextMenu
                ContextMenu {
                    MenuItem {
                        objectName: "DetailsmenuItem"

                        //% "Details"
                        text: qsTrId("filemanager-me-details")
                        onClicked: {
                            pageStack.animatorPush("DetailsPage.qml", {
                                                       fileName: model.fileName,
                                                       mimeType: model.mimeType,
                                                       isDir: model.isDir,
                                                       modified: model.modified,
                                                       size: model.size,
                                                       path: model.absolutePath
                                                   })
                        }
                    }

                    MenuItem {
                        objectName: "RenameMenuItem"

                        //% "Rename"
                        text: qsTrId("filemanager-me-rename")
                        visible: standardPaths.indexOf(model.absolutePath) == -1
                        onClicked: pageStack.animatorPush("RenameDialog.qml", { oldPath: model.absolutePath })
                    }

                    MenuItem {
                        objectName: "CutMenuItem"

                        visible: standardPaths.indexOf(model.absolutePath) == -1
                        //% "Cut"
                        text: qsTrId("filemanager-me-cut")
                        onClicked: FileEngine.cutFiles([ fileModel.fileNameAt(model.index) ])
                    }

                    MenuItem {
                        objectName: "CopyMenuItem"

                        //% "Copy"
                        text: qsTrId("filemanager-me-copy")
                        onClicked: FileEngine.copyFiles([ fileModel.fileNameAt(model.index) ])
                    }

                    MenuItem {
                        objectName: "GetPasteTextIDMenuItem"

                        readonly property string fullDirectoryPath: fileModel.path + "/" + fileName

                        text: _getPasteTextId()
                        visible: FileEngine.clipboardCount > 0 && model.isDir && FileEngine.isUserDir(fullDirectoryPath)
                        onClicked: FileEngine.pasteFiles(fullDirectoryPath, true, indexIfExists)
                    }

                    MenuItem {
                        objectName: "ShareMenuItem"

                        visible: !model.isDir && !model.isLink
                        //% "Share"
                        text: qsTrId("filemanager-me-share")
                        onClicked: {
                            shareAction.resources = [FileManager.pathToUrl(model.absolutePath)]
                            shareAction.trigger()
                        }

                        ShareAction {
                            id: shareAction
                        }
                    }

                    ExtractContextMenuItem {
                        objectName: "DirectoryPageExtractHereMenuItem"

                        archiveFile: model.absolutePath
                        extractorView: extractor
                        extractAllFiles: true
                        visible: model.isArchive && standardPaths.indexOf(model.absolutePath) == -1
                    }

                    ExtractContextMenuItem {
                        objectName: "DirectoryPageExtractToMenuItem"

                        archiveFile: model.absolutePath
                        extractorView: extractor
                        needSelectDirectory: true
                        extractAllFiles: true
                        visible: model.isArchive && standardPaths.indexOf(model.absolutePath) == -1
                    }

                    CompressContextMenuItem {
                        objectName: "DirectoryPageCompressMenuItem"

                        extractorView: extractor
                        files: [model.absolutePath]
                        path: page.path
                        visible: standardPaths.indexOf(model.absolutePath) == -1
                    }

                    MenuItem {
                        objectName: "DeleteMenuItem"

                        visible: standardPaths.indexOf(model.absolutePath) == -1
                        //% "Delete"
                        text: qsTrId("filemanager-me-delete")
                        onClicked: remove()
                    }
                }
            }
        }
        ViewPlaceholder {
            id: viewPlaceholder
            objectName: "NoFilesPlaceholder"

            enabled: (fileModel.count === 0 && fileModel.populated) || errorType !== FileModel.NoError
            visible: enabled
            text: errorType === FileModel.NoError ?
                      //% "No files"
                      qsTrId("filemanager-la-no_files") :
                      //% "No permissions to access %1"
                      qsTrId("filemanager-la-folder_no_permission_to_access").arg(fileModel.path)
        }
        VerticalScrollDecorator {}
    }

    Component {
        id: errorNotificationComponent

        FileManagerNotification {
            property var connections: Connections {
                target: FileEngine
                onError: {
                    switch (error) {
                    case FileEngine.ErrorOperationInProgress:
                        //% "File operation already in progress"
                        show(qsTrId("filemanager-la-operation_already_in_progress"))
                        break
                    case FileEngine.ErrorCopyFailed:
                        //% "Copying failed"
                        show(qsTrId("filemanager-la-copy_failed"))
                        break
                    case FileEngine.ErrorDeleteFailed:
                        //% "Deletion failed"
                        show(qsTrId("filemanager-la-deletion_failed"))
                        break
                    case FileEngine.ErrorMoveFailed:
                        //% "Moving failed"
                        show(qsTrId("filemanager-la-moving_failed"))
                        break
                    case FileEngine.ErrorRenameFailed:
                        //% "Renaming failed"
                        show(qsTrId("filemanager-la-renaming_failed"))
                        break
                    case FileEngine.ErrorCannotCopyIntoItself:
                        //: Shown both when trying to copy file over itself and copying a folder into itself
                        //% "You cannot copy into itself"
                        show(qsTrId("filemanager-la-cannot_copy_into_itself"))
                        break
                    case FileEngine.ErrorCannotMoveIntoItself:
                        //: Shown both when trying to move file over itself and moving a folder into itself
                        //% "You cannot move into itself"
                        show(qsTrId("filemanager-la-cannot_move_into_itself"))
                        break
                    case FileEngine.ErrorFolderCopyFailed:
                        //% "Copying folder failed"
                        show(qsTrId("filemanager-la-copying_folder_failed"))
                        break
                    case FileEngine.ErrorFolderCreationFailed:
                        //% "Could not create folder"
                        show(qsTrId("filemanager-la-folder_creation_failed"))
                        break
                    case FileEngine.ErrorChmodFailed:
                        //% "Could not set permission"
                        show(qsTrId("filemanager-la-set_permissions_failed"))
                        break
                    }
                    _deletingPath = "" // reset state if deletion failed
                }
                onModeChanged: {
                    if (FileEngine.mode === FileEngine.IdleMode) {
                        _deletingPath = ""
                    }
                }
            }
            property var busyView: BusyView {
                busy: FileEngine.busy
                enabled: busy || page.mounting
                text: {
                    if (page.mounting) {
                        //% "Mounting SD card"
                        return qsTrId("filemanager-la-mounting")
                    } else switch (FileEngine.mode) {
                        case FileEngine.DeleteMode:
                            //% "Deleting"
                            return qsTrId("filemanager-la-deleting")
                        case FileEngine.CopyMode:
                            //% "Copying"
                            return qsTrId("filemanager-la-copying")
                        default:
                            return ""
                    }
                }
            }
        }
    }

    ExtractorView {
        id: extractor
        objectName: "DirectoryPageExtractorView"

        onShowInfo: {
            if (!errorNotification) {
                errorNotification = errorNotificationComponent.createObject(page)
            }
            errorNotification.show(info)
        }
    }
}
