/*
 * Copyright (c) 2022 Open Mobile Platform LLC.
 *
 * License: BSD
 *
 * This file uses parts of the Qt Graphical Effects module which is
 * licensed under the BSD license.
 * You may use this file under the terms of the BSD license as follows:
 *
 * "Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *   * Neither the name of The Qt Company Ltd nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
*/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0

SequenceFilterPrivate {
    id: blur

    property bool isLightScheme: Theme.colorScheme === Theme.DarkOnLight

    ResizeFilterPrivate {
        size {
            width: 48
            height: 48
        }
        fillMode: FillPrivate.PreserveAspectFit
    }

    ShaderFilterPrivate {
        id: shaderFilter

        property real isLightScheme: blur.isLightScheme ? 1.0 : 0.0 // no bools in uniform_p.h

        fragmentShader: "
            uniform lowp sampler2D sourceTexture;
            uniform lowp float isLightScheme;

            varying highp vec2 sourceCoord;

            lowp vec3 RGBtoHSL(lowp vec3 color) {
                lowp float cmin = min(color.r, min(color.g, color.b));
                lowp float cmax = max(color.r, max(color.g, color.b));
                lowp float h = .0;
                lowp float s = .0;
                lowp float l = (cmin + cmax) / 2.;
                lowp float diff = cmax - cmin;
                if (diff > 1. / 256.) {
                    if (l < .5)
                        s = diff / (cmin + cmax);
                    else
                        s = diff / (2. - (cmin + cmax));
                    if (color.r == cmax)
                        h = (color.g - color.b) / diff;
                    else if (color.g == cmax)
                        h = 2. + (color.b - color.r) / diff;
                    else
                        h = 4. + (color.r - color.g) / diff;
                    h /= 6.;
                }
                return vec3(h, s, l);
            }

            lowp float hueToIntensity(lowp float v1, lowp float v2, lowp float h) {
                h = fract(h);
                if (h < 1. / 6.)
                    return v1 + (v2 - v1) * 6.0 * h;
                else if (h < 1. / 2.)
                    return v2;
                else if (h < 2. / 3.)
                    return v1 + (v2 - v1) * 6.0 * (2. / 3. - h);
                return v1;
            }

            lowp vec3 HSLtoRGB(lowp vec3 color) {
                lowp float h = color.x;
                lowp float l = color.z;
                lowp float s = color.y;
                if (s < 1. / 256.)
                    return vec3(l, l, l);
                lowp float v1;
                lowp float v2;
                if (l < 0.5)
                    v2 = l * (1. + s);
                else
                    v2 = (l + s) - (s * l);
                v1 = 2. * l - v2;
                lowp float d = 1.0 / 3.0;
                lowp float r = hueToIntensity(v1, v2, h + d);
                lowp float g = hueToIntensity(v1, v2, h);
                lowp float b = hueToIntensity(v1, v2, h - d);
                return vec3(r, g, b);
            }

            lowp float channelBlendHardLight(lowp float c1, lowp float c2) {
                return c2 > .5 ? (1. - (1. - 2. * (c2 - .5)) * (1. - c1)) : (2. * c1 * c2);
            }

            void main() {
                lowp vec4 texColor = texture2D(sourceTexture, sourceCoord);

                lowp vec3 hsl = RGBtoHSL(texColor.rgb);

                lowp float excludeLum = hsl.g + .7 - 2. * hsl.g * .7;
                lowp float differenceLum = abs(hsl.g - .9);

                lowp vec3 hardLight = vec3(0., 0., channelBlendHardLight(mix(excludeLum, differenceLum, isLightScheme), mix(.4, .8, isLightScheme)));
                lowp vec3 hardLightRGB = HSLtoRGB(hardLight);

                lowp vec3 mixed = mix(texColor.rgb * hardLightRGB, // Multiply
                                      vec3(1.) - (vec3(1.) - texColor.rgb) * (vec3(1.) - hardLightRGB), // Screen
                                      isLightScheme);

                gl_FragColor = vec4(mixed, 1.);
            }
            "
    }

    ConvolutionFilterPrivate {
        id: convolutionFilter

        kernel: KernelPrivate.gaussian(KernelPrivate.SampleSize17, 5)
    }
}
