/****************************************************************************
**
** Copyright (C) 2022 - 2023 Open Mobile Platform LLC.
**
****************************************************************************/

import QtQuick 2.0
import Nemo.DBus 2.0
import Sailfish.Silica 1.0
import ru.omp.printer 1.0

Dialog {
    id: root

    onAccepted: {
        var success = true
        if (currentPageRange.length != 0) {
            success = printerInfo.setPageRange(currentPrinter, currentPageRange)
        }

        if (success && scaleToPage) {
            success = printerInfo.setScaling(currentPrinter, true)
        }

        if (success) {
            success = printerInfo.setFormat(currentPrinter, currentFormat)
        }

        if (success) {
            for (var i = 0; i < numberOfCopies; ++i) {
                success = printerInfo.printFile(currentPrinter, printPath)
            }
        }

        if (!success || printerInfo.error) {
            pageStack.replace("PrintErrorPage.qml", { error: printerInfo.error })
        }
    }

    property string printPath
    property string currentPrinter: addedPrinters[0]
    property string currentFormat: formats[0]
    property var currentOrientation: PrinterInfo.Portrait
    property string currentPageRange
    property int numberOfCopies: 1
    property bool scaleToPage: paperSizePrintSwitch.active

    readonly property alias addedPrinters: printerInfo.addedPrinters
    readonly property var formats: printerInfo.availableFormats(currentPrinter)
    readonly property bool isPhone: Screen.sizeCategory <= Screen.Medium

    canAccept: addedPrinters.length !== 0

    PrinterInfo {
        id: printerInfo
    }

    DBusInterface {
        id: settingsDBus

        service: "com.jolla.settings"
        path: "/com/jolla/settings/ui"
        iface: "com.jolla.settings.ui"
    }

    SilicaFlickable {
        width: parent.width
        height: parent.height - Theme.paddingLarge * 2
        contentHeight: column.height

        Column {
            id: column

            width: parent.width
            visible: printerInfo.addedPrinters.count !== 0

            DialogHeader {
                //% "Print"
                acceptText: qsTrId("components_pickers-he-print")
                //% "Cancel"
                cancelText: qsTrId("components_pickers-he-cancel")
            }

            Item { width: 1; height: Theme.paddingMedium }

            ViewPlaceholder {
                enabled: !root.canAccept
                //% "No added printers found"
                text: qsTrId("components_pickers-he-no_added_printers")
                //% "Go to settings to add printer"
                hintText: qsTrId("components_pickers-he-extended_settings")
            }

            ComboBox {
                id: printerSelector

                //% "Printer"
                label: qsTrId("components_pickers-la-printer")
                visible: root.canAccept
                currentIndex: 0

                menu: ContextMenu {
                    Repeater {
                        model: addedPrinters
                        MenuItem {
                            onClicked: currentPrinter = modelData

                            textFormat: Text.StyledText
                            color: highlighted ? Theme.highlightColor : Theme.primaryColor
                            text: modelData
                        }
                    }
                }
            }

            SectionHeader {
                //% "Printer settings"
                text: qsTrId("components_pickers-la-printer-settings")
                horizontalAlignment: Text.AlignLeft
                visible: root.canAccept
            }

            TextField {
                text: "1"
                //% "Number of copies"
                label: qsTrId("libprinter-qml-plugin-la-number-of-copies")
                inputMethodHints: Qt.ImhFormattedNumbersOnly
                visible: root.canAccept
            }

            ComboBox {
                //% "Page range"
                label: qsTrId("components_pickers-la-page-range")
                visible: root.canAccept

                menu: ContextMenu {
                    MenuItem {
                        onClicked: {
                            if (pageSelection.visible) {
                                pageSelection.visible = false
                            }
                            currentPageRange = ""
                        }

                        textFormat: Text.StyledText
                        color: highlighted ? Theme.highlightColor : Theme.primaryColor
                        //% "All"
                        text: qsTrId("components_pickers-la-all")
                    }
                    MenuItem {
                        onClicked: {
                            if (pageSelection.visible) {
                                pageSelection.visible = false
                            }
                            currentPageRange = "1"
                        }

                        textFormat: Text.StyledText
                        color: highlighted ? Theme.highlightColor : Theme.primaryColor
                        //% "Current"
                        text: qsTrId("components_pickers-la-current")
                    }
                    MenuItem {
                        onClicked: {
                            pageSelection.visible = true
                            pageSelection.forceActiveFocus()
                        }

                        textFormat: Text.StyledText
                        color: highlighted ? Theme.highlightColor : Theme.primaryColor
                        //% "Select Pages"
                        text: qsTrId("components_pickers-la-select-pages")
                    }
                }
            }

            TextField {
                id: pageSelection

                onTextChanged: currentPageRange = text
                visible: false
                //% "Specify the range of pages (for example 1-5) or list the pages required for printing separated by commas (for example 2, 6, 18)"
                label: qsTrId("printer-qml-plugin-la-page-selection-description")
                inputMethodHints: Qt.ImhFormattedNumbersOnly
            }

            ValueButton {
                onClicked: {
                    const dialog = pageStack.push("PageSizePickerDialog.qml",
                        {
                            formats: root.formats,
                            currentFormat: root.currentFormat
                        })

                    dialog.formatClicked.connect(function(format) {
                        value = format
                        root.currentFormat = value
                    })
                }
                //% "Paper size"
                label: qsTrId("components_pickers-la-paper-size")
                value: currentFormat
                visible: root.canAccept
            }

            TextSwitch {
                id: twoSidePrintSwitch

                onClicked: paperSizePrintSwitch.checked = false
                //% "Two-side print"
                text: qsTrId("components_pickers-la-two-side-print")
                description: enabled ?
                    //% "If supported by printer"
                    qsTrId("components_pickers-ds-if-supported") :
                    //% "Not supported by printer"
                    qsTrId("components_pickers-ds-not-supported")
                visible: root.canAccept
                enabled: printerInfo.duplexModeSupported
                checked: false
            }

            TextSwitch {
                id: paperSizePrintSwitch

                onClicked: twoSidePrintSwitch.checked = false
                //% "Scaling to paper size"
                text: qsTrId("components_pickers-la-scaling-to-paper-size")
                visible: root.canAccept
                checked: true
            }

            ComboBox {
                //% "Orientation"
                label: qsTrId("components_pickers-la-orientation")
                visible: root.canAccept

                menu: ContextMenu {
                    MenuItem {
                        onClicked: currentOrientation = PrinterInfo.Portrait
                        textFormat: Text.StyledText
                        color: highlighted ? Theme.highlightColor : Theme.primaryColor
                        //% "Portrait"
                        text: qsTrId("components_pickers-la-portrait")
                    }
                    MenuItem {
                        onClicked: currentOrientation = PrinterInfo.Landscape
                        textFormat: Text.StyledText
                        color: highlighted ? Theme.highlightColor : Theme.primaryColor
                        //% "Landscape"
                        text: qsTrId("components_pickers-la-landscape")
                    }
                }
            }

            Item {
                width: 1
                height: Theme.paddingLarge
            }

            Button {
                onClicked: settingsDBus.call("showAddPrinterDialog")
                anchors.horizontalCenter: parent.horizontalCenter
                visible: canAccept
                preferredWidth: isPortrait && isPhone ? Theme.buttonWidthMedium : parent.width - Theme.horizontalPageMargin * 2
                //% "Extended settings"
                text: qsTrId("components_pickers-bt-extendended-settings")
            }
        }

        VerticalScrollDecorator {}
    }

    Component.onCompleted: printerInfo.setFormat(currentPrinter, currentFormat)
}
