/*
 * Copyright (c) 2022 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */

#ifndef SAILFISH_MDM_SYSTEMUPDATE_H
#define SAILFISH_MDM_SYSTEMUPDATE_H

#include <QDateTime>
#include <QObject>

namespace Sailfish {
namespace Mdm {

class SystemUpdatePrivate;
class Q_DECL_EXPORT SystemUpdate : public QObject
{
    Q_OBJECT

public:
    enum class Error {
        // Error mappings from the SystemUpdateManager namespace
        NoError = 0,
        IOError,
        NetworkError,
        MetadataError,
        PackageError,
        DBusError,
        SsuError,
        NotSupportedError,
        InternalError,
        BusyError,
        AuthenticationError,
        LowBattery,
        LowDisk,
        // Additional errors for MDM client
        NotAvailable = 100,
        ScheduleError,

        UnknownError = -1,
    };
    Q_ENUM(Error)

    enum class Method {
        Any = 0,
        PackageUpdate = 1,
        ResetUpdate = 2
    };
    Q_ENUM(Method)

    Q_PROPERTY(bool ready READ ready NOTIFY readyChanged)
    Q_PROPERTY(bool scheduled READ scheduled NOTIFY scheduledChanged)
    Q_PROPERTY(int progress READ progress NOTIFY progressChanged)
    Q_PROPERTY(QString version READ version NOTIFY progressChanged)
    Q_PROPERTY(Method method READ method NOTIFY progressChanged)
    Q_PROPERTY(qlonglong size READ size NOTIFY progressChanged)
    Q_PROPERTY(QDateTime lastChecked READ lastChecked NOTIFY progressChanged)

    explicit SystemUpdate(QObject *parent = nullptr);

    QString version() const;
    Method method() const;
    qlonglong size() const;
    QDateTime lastChecked() const;

    bool ready() const;
    bool scheduled() const;
    int progress() const;

public slots:
    void downloadUpdate(const QString &version, Method method, const QString &vendorDataUrl = {});
    void scheduleUpdate(QDateTime from, QDateTime to);
    void installUpdate();
    void cancelDownload();
    void cancelUpdate();

signals:
    void readyChanged();
    void scheduledChanged();
    void progressChanged();
    void versionChanged();
    void methodChanged();
    void sizeChanged();
    void lastCheckedChanged();
    void error(Error error, const QString &details);

private:
    SystemUpdatePrivate *d_ptr;
    Q_DISABLE_COPY(SystemUpdate)
    Q_DECLARE_PRIVATE(SystemUpdate)
};

}
}

#endif // SAILFISH_MDM_SYSTEMUPDATE_H
