/*
 * Copyright (c) 2020 - 2022 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */
import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0

Dialog {
    id: dialog
    objectName: "RenameDialog"

    property alias oldPath: fileInfo.file

    readonly property string _newPath: fileName.text !== "" && !_hasInvalidCharacters
            ? fileInfo.directoryPath + "/" + fileName.text
            : ""

    readonly property var _regExp: new RegExp("[\/*\?<>\|]+")
    readonly property var _regExpExtension: new RegExp(/\.[0-9a-z]+$/i)
    readonly property bool _hasInvalidCharacters: _regExp.test(fileName.text)
    readonly property bool _exists: _newPath !== "" && _newPath !== oldPath && FileEngine.exists(_newPath)
    readonly property bool _hasFirstSpace: _newPath !== "" && /^\s+/.test(fileName.text)
    readonly property bool _hasFirstDot: _newPath !== "" && !fileInfo.isDir
                                         ? fileName.text.split(_regExpExtension)[0].charAt(0) === "."
                                         : fileName.text.charAt(0) === "."
    readonly property bool _hasEmptyName: {
        var value = !fileInfo.isDir
        var matches = fileName.text.match(_regExpExtension)

        if (_newPath !== "" && (matches || []).length > 0)
            value = matches[0].length === fileName.text.length && !fileInfo.isDir

        return value
    }

    function _suffixForFileName(fileName) {
        var suffix = FileEngine.extensionForFileName(fileName)
        return suffix !== "" ?  "." + suffix : suffix
    }

    FileInfo {
        id: fileInfo
    }

    DialogHeader {
        id: dialogHeader
        objectName: "RenameDialogHeader"

        //% "Rename"
        title: qsTrId("filemanager-he-rename")
    }

    TextField {
        id: fileName
        objectName: "FileNameTextField"

        width: parent.width
        anchors.top: dialogHeader.bottom
        label: {
            if (dialog._hasInvalidCharacters || dialog._hasFirstDot) {
               return fileInfo.isDir
                        ? qsTrId("filemanager-te-invalid_foldername")
                            //% "Invalid file name"
                        : qsTrId("filemanager-te-invalid_filename")
            } else if (dialog._exists) {
                return fileInfo.isDir
                        ? qsTrId("filemanager-te-foldername_exists")
                            //% "A file with the same name exists"
                        : qsTrId("filemanager-te-filename_exists")
            } else if (dialog._hasFirstSpace) {
                return fileInfo.isDir
                        ? qsTrId("filemanager-te-foldername_can`t_start_with_spaces")
                            //% "File name can`t start with spaces"
                        : qsTrId("filemanager-te-filename_can`t_start_with_spaces")
            } else {
               //% "Title"
               return qsTrId("filemanager-la-title")
            }
        }

        placeholderText: qsTrId("filemanager-la-title")
        onFocusChanged: {
            if (focus) {
                var suffix = _suffixForFileName(text)

                select(0, text.length - suffix.length)
            }
        }

        text: fileInfo.fileName
        errorHighlight: dialog._hasInvalidCharacters || dialog._exists || dialog._hasFirstSpace || dialog._hasFirstDot

        EnterKey.iconSource: "image://theme/icon-m-enter-accept"
        EnterKey.enabled: text !== ""
        EnterKey.onClicked: accept()

        Component.onCompleted: {
            focus = true
        }
    }

    canAccept: _newPath !== "" && !_exists && !_hasFirstSpace && !_hasEmptyName && !_hasFirstDot
    onAccepted: {
        if (_newPath !== oldPath) {
            FileEngine.rename(oldPath, _newPath)
        }
    }
}
