#!/bin/bash
#
# Manage virtual repository with user installed toolings and targets
#
# Copyright (C) 2018 Jolla Ltd.
# Contact: Martin Kampas <martin.kampas@jolla.com>
# All rights reserved.
#
# You may use this file under the terms of BSD license as follows:
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#   * Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#   * Neither the name of the Jolla Ltd nor the
#     names of its contributors may be used to endorse or promote products
#     derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

set -o nounset
set -o pipefail

# no `readlink -f` on macOS
SELF=$(cd "$(dirname "$0")"; echo "$PWD/$(basename "$0")")

synopsis()
{
    cat <<END
usage: manage.sh [options] init
       manage.sh [options] touch <type> <name> <size>
       manage.sh [options] rm <type> <name>
END
}

short_usage()
{
    cat <<END
$(synopsis)

Try 'manage.sh --help' for more information.
END
}

usage()
{
    less --quit-if-one-screen <<END
$(synopsis)

Manage virtual repository with user installed toolings and targets

COMMANDS
    init
        Initialize an empty repository and copy itself into the repository

    touch <type> <name> <size>
        Touch a component representing the object of the given <type>
        ('target' or 'tooling'), <name> and <size> in bytes. If the component
        does not exist, it will be created. If the component already exists,
        the data will be updated.

    rm <type> <name>
        Remove a component representing the object of the given <type> and
        <name> if it exists. Does nothing if the object does not exist.

GLOBAL OPTIONS
    -r, --repository PATH
        Repository path to operate on. Defaults to the current working directory
END
}

fatal()
{
    printf 'manage.sh: fatal: %s\n' "$*" >&2
}

warning()
{
    printf 'manage.sh: warning: %s\n' "$*" >&2
}

bad_usage()
{
    fatal "$*"
    short_usage >&2
}

with_tmp_file()
{
    local file=$1 cmd=("${@:2}")
    local tmp_file=

    with_tmp_file_cleanup()
    (
        trap 'echo cleaning up...' INT TERM HUP
        if [[ $tmp_file ]]; then
            rm -f "$tmp_file"
        fi
    )
    trap 'with_tmp_file_cleanup; trap - RETURN' RETURN
    trap 'return 1' INT TERM HUP

    tmp_file=$(mktemp "$file.XXX") || return

    if "${cmd[@]}" <&3 >"$tmp_file"; then
        cat <"$tmp_file" >"$file" || return
    else
        return $?
    fi
} 3<&0 <&-

xml_begin()
{
    cat <<END
<Updates>
    $XML_PREAMBLE
END
}

today_version()
{
    date "+%Y.%m.%d"
}

today_date()
{
    date "+%Y-%m-%d"
}

escape_name()
{
    local name=$1
    name=${name//_/_5F}
    name=${name//-/_2D}
    name=${name//./_2E}
    echo "$name"
}

target_component_name()
{
    local name=$1
    echo "$TARGETS_CONTAINER_NAME.$(escape_name "$name")"
}

tooling_component_name()
{
    local name=$1
    echo "$TOOLINGS_CONTAINER_NAME.$(escape_name "$name")"
}

xml_targets_container()
{
    cat <<END
    <PackageUpdate>
        <Name>$TARGETS_CONTAINER_NAME</Name>
        <Version>$(today_version)</Version>
        <ReleaseDate>$(today_date)</ReleaseDate>
        <DisplayName>$TARGETS_CONTAINER_DISPLAY_NAME</DisplayName>
        <Description>$TARGETS_CONTAINER_DESCRIPTION</Description>
        <SortingPriority>-2</SortingPriority>
        <Checkable>false</Checkable>
        <UpdateFile CompressedSize="0" OS="Any" UncompressedSize="0"/>
    </PackageUpdate>
END
}

xml_toolings_container()
{
    cat <<END
    <PackageUpdate>
        <Name>$TOOLINGS_CONTAINER_NAME</Name>
        <Version>$(today_version)</Version>
        <ReleaseDate>$(today_date)</ReleaseDate>
        <DisplayName>$TOOLINGS_CONTAINER_DISPLAY_NAME</DisplayName>
        <Description>$TOOLINGS_CONTAINER_DESCRIPTION</Description>
        <SortingPriority>-1</SortingPriority>
        <ForcedInstallation>true</ForcedInstallation>
        <Virtual>true</Virtual>
        <UpdateFile CompressedSize="0" OS="Any" UncompressedSize="0"/>
    </PackageUpdate>
END
}

xml_target_component()
{
    local target_name=$1 size=$2

    cat <<END
    <PackageUpdate>
        <Name>$(target_component_name "$target_name")</Name>
        <Version>$(today_version)</Version>
        <ReleaseDate>$(today_date)</ReleaseDate>
        <DisplayName>$target_name</DisplayName>
        <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
        <AutoDependOn>$AUTO_DEPEND_ON</AutoDependOn>
        <UpdateFile CompressedSize="0" OS="Any" UncompressedSize="$size"/>
    </PackageUpdate>
END
}

xml_tooling_component()
{
    local tooling_name=$1 size=$2

    cat <<END
    <PackageUpdate>
        <Name>$(tooling_component_name "$tooling_name")</Name>
        <Version>$(today_version)</Version>
        <ReleaseDate>$(today_date)</ReleaseDate>
        <DisplayName>$tooling_name</DisplayName>
        <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
        <AutoDependOn>$AUTO_DEPEND_ON</AutoDependOn>
        <Virtual>true</Virtual>
        <UpdateFile CompressedSize="0" OS="Any" UncompressedSize="$size"/>
    </PackageUpdate>
END
}

xml_end()
{
    cat <<END
</Updates>
END
}

xpath()
{
    local xml=$1 xpath=$2

    [[ ! $xml ]] && return

    # On empty match, xmllint says "XPath set is empty" on stderr and exits with non-zero
    local stderr=
    { stderr=$(xmllint --xpath "$xpath" - <<<"$xml" 3>&1 1>&2 2>&3 3>&-); } 2>&1
    local rc=$?

    if [[ $rc -eq 0 || $rc -eq $XPATH_SET_IS_EMPTY_RC ]]; then
        return 0
    else
        printf "%s\n" "$stderr" >&2
        return $rc
    fi
}

xml_pass_targets_container()
{
    local xml=$1
    xpath "$xml" "/Updates/PackageUpdate[Name/text()='$TARGETS_CONTAINER_NAME']"
}

xml_pass_toolings_container()
{
    local xml=$1
    xpath "$xml" "/Updates/PackageUpdate[Name/text()='$TOOLINGS_CONTAINER_NAME']" 
}

xml_pass_other_target_components()
{
    local xml=$1 object_name=$2
    local name=$(target_component_name "$object_name")
    xpath "$xml" "/Updates/PackageUpdate[starts-with(Name/text(), '$TARGETS_CONTAINER_NAME.')
                                         and not(starts-with(Name/text(), '$TOOLINGS_CONTAINER_NAME.'))
                                         and Name/text() != '$TOOLINGS_CONTAINER_NAME'
                                         and Name/text() != '$name']"
}

xml_pass_other_tooling_components()
{
    local xml=$1 object_name=$2
    local name=$(tooling_component_name "$object_name")
    xpath "$xml" "/Updates/PackageUpdate[starts-with(Name/text(), '$TOOLINGS_CONTAINER_NAME.')
                                         and Name/text() != '$name']"
}

xml_touch_object_()
{
    local type=$1 name=$2 size=$3

    local xml=
    read -d '' -r xml

    {
        xml_begin

        if [[ $type == "$TYPE_TARGET" ]]; then
            xml_targets_container
            xml_pass_toolings_container "$xml" || return
            xml_target_component "$name" "$size"
            xml_pass_other_target_components "$xml" "$name" || return
            xml_pass_other_tooling_components "$xml" "" || return
        else
            local targets_container=
            targets_container=$(xml_pass_targets_container "$xml") || return
            if [[ $targets_container ]]; then
                printf "%s\n" "$targets_container"
            else
                xml_targets_container
            fi
            xml_toolings_container
            xml_pass_other_target_components "$xml" "" || return
            xml_tooling_component "$name" "$size"
            xml_pass_other_tooling_components "$xml" "$name" || return
        fi

        xml_end
    } |xmllint --format -
}

xml_touch_object()
{
    touch "$UPDATES_XML" || return
    with_tmp_file "$UPDATES_XML" xml_touch_object_ "$@" <"$UPDATES_XML"
}

xml_rm_object_()
{
    local type=$1 name=$2

    local xml=
    read -d '' -r xml

    {
        xml_begin

        local old_target_components= target_components=
        local old_tooling_components= tooling_components=

        if [[ $type == "$TYPE_TARGET" ]]; then
            old_target_components=$(xml_pass_other_target_components "$xml" "") || return
            target_components=$(xml_pass_other_target_components "$xml" "$name") || return
            old_tooling_components=$(xml_pass_other_tooling_components "$xml" "") || return
            tooling_components=$old_tooling_components
        else
            old_target_components=$(xml_pass_other_target_components "$xml" "") || return
            target_components=$old_target_components
            old_tooling_components=$(xml_pass_other_tooling_components "$xml" "") || return
            tooling_components=$(xml_pass_other_tooling_components "$xml" "$name") || return
        fi

        if [[ $target_components || $tooling_components ]]; then
            if [[ $target_components != "$old_target_components" ]]; then
                xml_targets_container
            else
                xml_pass_targets_container "$xml" || return
            fi
        fi

        if [[ $tooling_components ]]; then
            if [[ $tooling_components != "$old_tooling_components" ]]; then
                xml_toolings_container
            else
                xml_pass_toolings_container "$xml" || return
            fi
        fi

        printf "%s\n" "$target_components"
        printf "%s\n" "$tooling_components"

        xml_end
    } |if ! which xmllint &>/dev/null; then
        # Allow use of the 'init' command without xmllint
        cat
    else
        xmllint --format -
    fi
}

xml_rm_object()
{
    touch "$UPDATES_XML" || return
    with_tmp_file "$UPDATES_XML" xml_rm_object_ "$@" <"$UPDATES_XML"
}

meta_update()
{
    local empty7z=
    read -d '' -r empty7z <<'END'
00000000: 377a bcaf 271c 0004 8d9b d50f 0000 0000  7z..'...........
00000010: 0000 0000 0000 0000 0000 0000 0000 0000  ................
END

    local xsl=
    read -d '' -r xsl <<'END'
<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">
  <xsl:output method="text" omit-xml-declaration="yes" indent="no"/>
  <xsl:template match="Updates">
    <xsl:for-each select="PackageUpdate">
      <xsl:value-of select="concat(Version, ' ', Name, '&#10;')"/>
    </xsl:for-each>
  </xsl:template>
</xsl:stylesheet>
END
    local packages=$(xsltproc <(printf '%s' "$xsl") "$UPDATES_XML") || return

    find -name '*meta.7z' -delete || return
    find -maxdepth 1 -type d -delete || return

    local version= package=
    ! [[ $packages ]] || while read version package; do
        mkdir "$package" || return
        xxd -r <<<"$empty7z" >"$package/${version}meta.7z" || return
    done <<<"$packages"
}

set_defaults()
{
    UPDATES_XML=Updates.xml
    APPLICATION_NAME="Aurora SDK"
    APPLICATION_VERSION="2.0.0"
    XML_PREAMBLE="
<ApplicationName>$APPLICATION_NAME</ApplicationName>
<ApplicationVersion>$APPLICATION_VERSION</ApplicationVersion>
<Checksum>true</Checksum>"
    TARGETS_CONTAINER_NAME="org.merproject.targets.user"
    TARGETS_CONTAINER_DISPLAY_NAME="Custom Build Targets"
    TARGETS_CONTAINER_DESCRIPTION="Custom build targets managed via sfdk."
    TARGET_COMPONENT_DESCRIPTION="This is a custom build target managed via sfdk. It cannot be deselected here."
    TOOLINGS_CONTAINER_NAME="org.merproject.targets.user.toolings"
    TOOLINGS_CONTAINER_DISPLAY_NAME="Toolings"
    TOOLINGS_CONTAINER_DESCRIPTION="Custom build toolings managed via sfdk."
    TOOLING_COMPONENT_DESCRIPTION="This is a custom build tooling managed via sfdk. It cannot be deselected here."
    AUTO_DEPEND_ON="org.merproject.mersdk"

    XPATH_SET_IS_EMPTY_RC=$(xmllint --xpath "/Bar" - <<<"<Foo/>" 2>/dev/null; echo $?)

    TYPE_TARGET=target
    TYPE_TOOLING=tooling

    OPT_REPOSITORY=$PWD
    OPT_TYPE=
    OPT_INIT=
    OPT_TOUCH=
    OPT_RM=
    OPT_OBJECT=
    OPT_SHORT_USAGE=
    OPT_USAGE=
}

parse_opts()
{
    local positional_args=()
    while (( $# > 0 )); do
        case $1 in
            -h)
                OPT_SHORT_USAGE=1
                return
                ;;
            --help)
                OPT_USAGE=1
                return
                ;;
            -r|--repository)
                if [[ ! ${2:-} ]]; then
                    bad_usage "Argument expected: '$1'"
                    return 1
                fi
                OPT_REPOSITORY=$2
                shift
                ;;
            -*)
                bad_usage "Unrecognized option: '$1'"
                return 1
                ;;
            *)
                positional_args+=("$1")
                ;;
        esac
        shift
    done

    set -- ${positional_args[@]:+"${positional_args[@]}"}

    if (( $# == 0 )); then
        bad_usage "Command expected"
        return 1
    fi

    case $1 in
        init)
            if (( $# != 1)); then
                bad_usage "init: Unexpected argument: '$2'"
                return 1
            fi
            OPT_INIT=1
            ;;
        touch)
            if (( $# != 4)); then
                bad_usage "touch: Exactly three arguments expected"
                return 1
            fi
            OPT_TOUCH=1
            OPT_TYPE=$2
            OPT_OBJECT=$3
            OPT_SIZE=$4
            if ! [[ $OPT_SIZE =~ ^[0-9]+$ ]]; then
                bad_usage "A decimal integer expected: '$OPT_SIZE'"
                return 1
            fi
            ;;
        rm)
            if (( $# != 3)); then
                bad_usage "rm: Exactly two arguments expected"
                return 1
            fi
            OPT_RM=1
            OPT_TYPE=$2
            OPT_OBJECT=$3
            ;;
        *)
            bad_usage "Unknown command: '$1'"
            return 1
            ;;
    esac

    if [[ ! $OPT_INIT ]]; then
        case $OPT_TYPE in
            "$TYPE_TOOLING"|"$TYPE_TARGET")
                :
                ;;
            *)
                bad_usage "Not a valid object type: '$OPT_TYPE'"
                return 1
                ;;
        esac
    fi
}

main()
{
    set_defaults || return
    parse_opts "$@" || return

    if [[ $OPT_SHORT_USAGE ]]; then
        short_usage
        return
    fi

    if [[ $OPT_USAGE ]]; then
        usage
        return
    fi

    cd "$OPT_REPOSITORY" || return

    if [[ $OPT_INIT ]]; then
        cp "$SELF" . || return
        xml_rm_object "$TYPE_TARGET" "nonsense" || return
        return
    fi

    if [[ $OPT_TOUCH ]]; then
        xml_touch_object "$OPT_TYPE" "$OPT_OBJECT" "$OPT_SIZE" || return
        meta_update || return
        return
    fi

    if [[ $OPT_RM ]]; then
        xml_rm_object "$OPT_TYPE" "$OPT_OBJECT" || return
        meta_update || return
        return
    fi
}

if [[ ${1:-} != --self-test ]]; then
    main "$@"
    exit
fi

##############################################################################
############  S E L F - T E S T  EXECUTION BEGINS HERE #######################
##############################################################################

: ${MANAGE_SH_SELF_TEST_VERBOSE:=}
tc_num=0
tc_failed_num=0

quote() {
    [[ $# -gt 0 ]] && printf "%q " "$@" |sed 's/ $//'
}

set_up_ts() {
    local ts=$1
    TS_NAME=$2
    ${ts}_ts_set_up "${@:3}"
    if [[ $? -ne 0 ]]; then
        fatal "Test suite set-up failed: $ts"
    fi
}

tear_down_ts() {
    local ts=$1
    ${ts}_ts_tear_down
    if [[ $? -ne 0 ]]; then
        fatal "Test suite tear-down failed: $ts"
    fi
    TS_NAME=
}

run_tc() {
    local tc=$1
    TC_NAME=$2
    local args=("${@:3}")

    let tc_num++
    echo "*** Executing test case: ${TS_NAME:+$TS_NAME - }$TC_NAME"

    local stderr=
    { stderr=$(${tc}_tc ${args[@]:+"${args[@]}"} 3>&1 1>&2 2>&3 3>&-); } 2>&1
    local rc=$?

    if [[ $rc -ne 0 ]]; then
        let tc_failed_num++
    fi

    if [[ $rc -ne 0 || $MANAGE_SH_SELF_TEST_VERBOSE ]]; then
        cat <<END
  ** Stderr     ** [[
$stderr
]]
END
    fi

    return $rc
}

xmlformat()
{
    local xml=
    read -d '' -r xml
    [[ ! $xml ]] || xmllint --format - <<<"$xml"
}

################################################################################
# Test managing Updates.xml

manage_updates_xml_tc()
{
    local commands=("${@:1:$#-1}")
    local expected=${@:$#:1}

    :> "$UPDATES_XML" || return

    local command=
    for command in "${commands[@]}"; do
        $command || return
    done

    local actual= actual= diff=
    read -d '' -r actual < "$UPDATES_XML"
    if ! diff=$(diff <(xmlformat <<<"$expected") <(xmlformat <<<"$actual")); then
        cat <<END
Test case failed: $TC_NAME
  ** Commands   ** [[
$(for command in "${commands[@]}"; do
    command=($command)
    quote "${command[@]}"
    echo
done)
]]
  ** Mismatch   ** [[
$diff
]]
  ** Expected   ** [[
$(xmlformat <<<"$expected")
]]
  ** Actual     ** [[
$(xmlformat <<<"$actual")
]]
  ** Actual RAW ** [[
$actual
]]
END
        return 1
    fi
}

manage_updates_xml_ts_set_up()
{
    set_defaults || return

    UPDATES_XML=$(mktemp "$UPDATES_XML.XXX") || return

    ts_with_date()
    {
        local ts_date=$1
        local command=("${@:2}")

        date()
        {
            command date "$@" --date="$ts_date 12:00"
        }

        "${command[@]}"
        local retv=$?

        unset date

        return $retv
    }

    ts_xml_targets_container()
    {
        local date=$1
        printf "%s\n" "
<PackageUpdate>
    <Name>$TARGETS_CONTAINER_NAME</Name>
    <Version>${date//-/.}</Version>
    <ReleaseDate>$date</ReleaseDate>
    <DisplayName>$TARGETS_CONTAINER_DISPLAY_NAME</DisplayName>
    <Description>$TARGETS_CONTAINER_DESCRIPTION</Description>
    <SortingPriority>-2</SortingPriority>
    <Checkable>false</Checkable>
    <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='0'/>
</PackageUpdate>"
    }

    ts_xml_toolings_container()
    {
        local date=$1
        printf "%s\n" "
<PackageUpdate>
    <Name>$TOOLINGS_CONTAINER_NAME</Name>
    <Version>${date//-/.}</Version>
    <ReleaseDate>$date</ReleaseDate>
    <DisplayName>$TOOLINGS_CONTAINER_DISPLAY_NAME</DisplayName>
    <Description>$TOOLINGS_CONTAINER_DESCRIPTION</Description>
    <SortingPriority>-1</SortingPriority>
    <ForcedInstallation>true</ForcedInstallation>
    <Virtual>true</Virtual>
    <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='0'/>
</PackageUpdate>"
    }
}

manage_updates_xml_ts_tear_down()
{
    unset ts_with_date
    unset ts_xml_targets_container
    unset ts_xml_toolings_container
    if [[ $UPDATES_XML ]]; then
        rm -f "$UPDATES_XML"
    fi
}

set_up_ts manage_updates_xml "Managing Updates.xml"

run_tc manage_updates_xml "Add target to empty" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object target Foo-1.0 20180101)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-01)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.01</Version>
            <ReleaseDate>2018-01-01</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180101'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Add tooling to empty" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-01)
        $(ts_xml_toolings_container 2018-01-01)
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.01</Version>
            <ReleaseDate>2018-01-01</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180101'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Remove last target" \
    "$(quote xml_touch_object target Foo-1.0 424242)" \
    "$(quote xml_rm_object target Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
    </Updates>"

run_tc manage_updates_xml "Remove last target, tooling exists" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object target Foo-1.0 20180102)" \
    "$(quote ts_with_date 2018-01-03 xml_rm_object target Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-03)
        $(ts_xml_toolings_container 2018-01-01)
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.01</Version>
            <ReleaseDate>2018-01-01</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180101'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Remove target" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object target Foo-1.0 20180102)" \
    "$(quote ts_with_date 2018-01-03 xml_touch_object target Bar-1.0 20180103)" \
    "$(quote ts_with_date 2018-01-04 xml_rm_object target Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-04)
        $(ts_xml_toolings_container 2018-01-01)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Bar_2D1_2E0</Name>
            <Version>2018.01.03</Version>
            <ReleaseDate>2018-01-03</ReleaseDate>
            <DisplayName>Bar-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180103'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.01</Version>
            <ReleaseDate>2018-01-01</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180101'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Remove last tooling" \
    "$(quote xml_touch_object tooling Foo-1.0 424242)" \
    "$(quote xml_rm_object tooling Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
    </Updates>"

run_tc manage_updates_xml "Remove last tooling, target exists" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object target Foo-1.0 20180102)" \
    "$(quote ts_with_date 2018-01-03 xml_rm_object tooling Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-02)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.02</Version>
            <ReleaseDate>2018-01-02</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180102'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Remove tooling" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object target Foo-1.0 20180102)" \
    "$(quote ts_with_date 2018-01-03 xml_touch_object tooling Bar-1.0 20180103)" \
    "$(quote ts_with_date 2018-01-04 xml_rm_object tooling Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-02)
        $(ts_xml_toolings_container 2018-01-04)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.02</Version>
            <ReleaseDate>2018-01-02</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180102'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Bar_2D1_2E0</Name>
            <Version>2018.01.03</Version>
            <ReleaseDate>2018-01-03</ReleaseDate>
            <DisplayName>Bar-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180103'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Update target size" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object target Foo-1.0 20180102)" \
    "$(quote ts_with_date 2018-01-03 xml_touch_object target Bar-1.0 20180103)" \
    "$(quote ts_with_date 2018-01-04 xml_touch_object target Foo-1.0 20180104)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-04)
        $(ts_xml_toolings_container 2018-01-01)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.04</Version>
            <ReleaseDate>2018-01-04</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180104'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Bar_2D1_2E0</Name>
            <Version>2018.01.03</Version>
            <ReleaseDate>2018-01-03</ReleaseDate>
            <DisplayName>Bar-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180103'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.01</Version>
            <ReleaseDate>2018-01-01</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180101'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Update tooling size" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object tooling Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object target Foo-1.0 20180102)" \
    "$(quote ts_with_date 2018-01-03 xml_touch_object tooling Bar-1.0 20180103)" \
    "$(quote ts_with_date 2018-01-04 xml_touch_object tooling Foo-1.0 20180104)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-02)
        $(ts_xml_toolings_container 2018-01-04)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.02</Version>
            <ReleaseDate>2018-01-02</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180102'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.04</Version>
            <ReleaseDate>2018-01-04</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180104'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Bar_2D1_2E0</Name>
            <Version>2018.01.03</Version>
            <ReleaseDate>2018-01-03</ReleaseDate>
            <DisplayName>Bar-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180103'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Remove non-existing object" \
    "$(quote ts_with_date 2018-01-01 xml_touch_object target Foo-1.0 20180101)" \
    "$(quote ts_with_date 2018-01-02 xml_touch_object tooling Foo-1.0 20180102)" \
    "$(quote ts_with_date 3018-01-02 xml_rm_object target Bar-1.0)" \
    "$(quote ts_with_date 3018-01-03 xml_rm_object tooling Bar-1.0)" \
    "<Updates>
        $XML_PREAMBLE
        $(ts_xml_targets_container 2018-01-01)
        $(ts_xml_toolings_container 2018-01-02)
        <PackageUpdate>
            <Name>$TARGETS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.01</Version>
            <ReleaseDate>2018-01-01</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TARGET_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180101'/>
        </PackageUpdate>
        <PackageUpdate>
            <Name>$TOOLINGS_CONTAINER_NAME.Foo_2D1_2E0</Name>
            <Version>2018.01.02</Version>
            <ReleaseDate>2018-01-02</ReleaseDate>
            <DisplayName>Foo-1.0</DisplayName>
            <Description>$TOOLING_COMPONENT_DESCRIPTION</Description>
            <AutoDependOn>org.merproject.mersdk</AutoDependOn>
            <Virtual>true</Virtual>
            <UpdateFile CompressedSize='0' OS='Any' UncompressedSize='20180102'/>
        </PackageUpdate>
    </Updates>"

run_tc manage_updates_xml "Remove from empty" \
    "$(quote ts_with_date 2018-01-01 xml_rm_object target Foo-1.0)" \
    "$(quote ts_with_date 2018-01-02 xml_rm_object tooling Foo-1.0)" \
    "<Updates>
        $XML_PREAMBLE
    </Updates>"

tear_down_ts manage_updates_xml

##############################################################################

if [[ $tc_failed_num -eq 0 ]]; then
    echo "*** All $tc_num tests passed"
else
    echo "*** $tc_failed_num out of $tc_num failed"
fi
