/****************************************************************************************
**
** SPDX-FileCopyrightText: 2011 Nokia Corporation and/or its subsidiary(-ies).
** SPDX-FileCopyrightText: 2013-2020 Jolla Ltd.
** SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.0
import QtQuick.Window 2.2 as QtQuick
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private

Private.SilicaMouseArea {
    id: page

    // The status of the page. One of the following:
    //      PageStatus.Inactive - the page is not visible
    //      PageStatus.Activating - the page is transitioning into becoming the active page
    //      PageStatus.Active - the page is the current active page
    //      PageStatus.Deactivating - the page is transitioning into becoming inactive
    readonly property int status: __stack_container
                                  ? __stack_container.status
                                  : (parent && parent.hasOwnProperty("__auroracontrols_splitview") && visible ? PageStatus.Active : PageStatus.Inactive)

    property bool backNavigation: true
    property bool showNavigationIndicator: true
    property bool forwardNavigation: _belowTop
    property bool canNavigateForward: forwardNavigation
    property int navigationStyle: PageNavigation.Horizontal
    readonly property bool _horizontalNavigationStyle: navigationStyle === PageNavigation.Horizontal
    property Item pageContainer
    property Item __stack_container

    property color backgroundColor: Theme.rgba(palette.overlayBackgroundColor, 0)
    property Component background: {
        if (backgroundColor.a !== 0) {
            return backgroundComponent
        } else if (pageContainer) {
            return pageContainer.pageBackground
        } else {
            return null
        }
    }

    property bool highContrast

    property int allowedOrientations: __silica_applicationwindow_instance._defaultPageOrientations
    property int orientation: orientationState.orientation

    property alias orientationTransitions: orientationState.transitions
    property alias defaultOrientationTransition: orientationState.defaultTransition
    property bool orientationTransitionRunning

    property bool isPortrait: (orientation === Qt.PortraitOrientation || orientation === Qt.InvertedPortraitOrientation)
                              || orientation === Qt.PrimaryOrientation && page.QtQuick.Screen.primaryOrientation === Qt.PortraitOrientation
    property bool isLandscape: (orientation === Qt.LandscapeOrientation || orientation === Qt.InvertedLandscapeOrientation)
                               || orientation === Qt.PrimaryOrientation && page.QtQuick.Screen.primaryOrientation === Qt.LandscapeOrientation

    readonly property int _navigation: __stack_container ? __stack_container.navigation : PageNavigation.NoNavigation
    readonly property int _navigationPending: __stack_container ? __stack_container.navigationPending : PageNavigation.NoNavigation
    readonly property int _direction: __stack_container ? __stack_container.direction : PageNavigation.NoDirection

    property int _allowedOrientations: {
        var allowed = allowedOrientations & __silica_applicationwindow_instance.allowedOrientations
        if (!allowed) {
            // No common supported orientations, let the page decide
            allowed = allowedOrientations
        }
        return allowed
    }
    property alias _windowOrientation: orientationState.pageOrientation

    property int _horizontalDimension: parent ? parent.width : page.QtQuick.Screen.width
    property int _verticalDimension: parent ? parent.height : page.QtQuick.Screen.height

    property var _forwardDestination
    property var _forwardDestinationProperties
    property int _forwardDestinationAction: PageStackAction.Push
    property Item _forwardDestinationInstance
    property var _forwardDestinationReplaceTarget
    property int _depth: parent && parent.hasOwnProperty("pageStackIndex") ? parent.pageStackIndex : -1

    property alias _windowOpacity: page.opacity

    property bool _opaqueBackground: background !== null && background != backgroundComponent
    readonly property bool _exposed:  pageContainer
                && __stack_container
                && pageContainer.visible
                && ((pageContainer._currentContainer === __stack_container)
                    || (pageContainer._currentContainer !== null && (pageContainer._currentContainer === __stack_container.transitionPartner)))
    readonly property bool _belowTop: pageContainer
                && __stack_container
                && pageContainer._currentContainer === __stack_container
                && __stack_container.attachedContainer !== null
    property bool _clickablePageIndicators: true
    property bool _isWindowPageStackContainer: true

    // The pending orientation state is used to avoid early changes the page geometry,
    // before recalculating all possible page margins, paddings or insets.
    property bool _orientationPending

    property int __silica_page
    readonly property Item __silica_page_instance: page

    function _orientationChangeDefaultAction() {
        // All orientation-related properties have been changed to their
        // relevant values, use default action for updating the SafeZoneRect
        // immediately and changing page geometry with new dimensions.
        SafeZoneRect.updateImmediately()
        _orientationPending = false
    }

    function _updateIsWindowPageStackContainer() {
        if (pageContainer) {
            _isWindowPageStackContainer = pageContainer.hasOwnProperty("__silica_window_pagestack")
        }
    }

    onOrientationChanged: {
        if (orientationState.defaultTransitionRunning) {
            return
        }
        if (orientationState.pageOrientation === orientation) {
            // Reset pending state if used no default transitions.
            _orientationPending = false
        }
    }

    onOrientationTransitionRunningChanged: {
        if (!orientationTransitionRunning) {
            // Reset pending state anyway when no transitions running.
            _orientationPending = false
        }
    }

    onPageContainerChanged: _updateIsWindowPageStackContainer()

    // This unusual binding avoids a warning when the page is destroyed.
    anchors.centerIn: page ? parent : null

    width: orientationState.width
    height: orientationState.height

    opacity: orientationTransitionRunning, 1
    scale: orientationTransitionRunning, 1

    rotation: orientationState.rotation
    visible: __stack_container && __stack_container.visible
    focus: true

    Component {
        id: backgroundComponent

        Rectangle {
            property color backgroundColor: page.backgroundColor

            onBackgroundColorChanged: {
                if (backgroundColor.a !== 0) {
                    color = backgroundColor
                }
            }
        }
    }

    StateGroup {
        id: dimensionState

        states: [
            State {
                when: !page.pageContainer || !page._exposed || !orientationState.completed
                      || page._orientationPending
                      || orientationState.pageOrientation === Qt.PrimaryOrientation
                      || orientationState.pageOrientation !== page.orientation
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    // Unbind previous values and hold them until the page
                    // is not exposed or not rotated to the desired orientation.
                    width: orientationState.width
                    height: orientationState.height
                }
            },
            State {
                name: 'Normal'
                when: orientationState.pageRotation % 180 == 0
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    width: page._horizontalDimension
                    height: page._verticalDimension
                }
            },
            State {
                name: 'Transposed'
                when: orientationState.pageRotation % 180 != 0
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    width: page._verticalDimension
                    height: page._horizontalDimension
                }
            }
        ]
    }

    StateGroup {
        id: orientationState

        property bool completed
        property bool defaultTransitionRunning

        // These values changes immediately
        // Choose the orientation this page will have given the current device orientation
        property int pageOrientation: Qt.PrimaryOrientation
        property int desiredPageOrientation: page._isWindowPageStackContainer
                                             ? __silica_applicationwindow_instance._selectOrientation(page._allowedOrientations, __silica_applicationwindow_instance.deviceOrientation)
                                             : page.QtQuick.Screen.primaryOrientation
        property bool desiredPageOrientationSuitable: !page._isWindowPageStackContainer
                                                      || (desiredPageOrientation & __silica_applicationwindow_instance.deviceOrientation)
        readonly property int pageRotation: page.QtQuick.Screen.angleBetween(pageOrientation,
                                                                             page.QtQuick.Screen.primaryOrientation)

        // These are the state managed orientation derived properties. Normally the pages properties
        // of the same name are bound to these but just prior to the orientation state changing those
        // bindings will be broken so property changes due to state fast-forwarding aren't propagated.
        property real width: rotation % 180 == 0 ? page._horizontalDimension : page._verticalDimension
        property real height: rotation % 180 == 0 ? page._verticalDimension : page._horizontalDimension

        // These values changes via transform animation
        property int orientation: Qt.PrimaryOrientation
        property real rotation

        onDesiredPageOrientationChanged: _updatePageOrientation()
        onDesiredPageOrientationSuitableChanged: _updatePageOrientation()

        function _updatePageOrientation() {
            if (pageOrientation !== desiredPageOrientation) {
                // Lock in the current values of the orientation properties, when the pageOrientation
                // property and the states which derive from it change the fast-forwarded values
                // will be the same as the
                page.orientation = page.orientation
                page.rotation = page.rotation
                page.width = page.width
                page.height = page.height
                page._orientationPending = true

                const init = pageOrientation === Qt.PrimaryOrientation
                pageOrientation = desiredPageOrientation

                // State fast forwarding is done, restore the bindings.
                page.orientation = Qt.binding(function() { return orientationState.orientation })
                page.rotation = Qt.binding(function() { return orientationState.rotation })
                page.width = Qt.binding(function() { return orientationState.width })
                page.height = Qt.binding(function() { return orientationState.height })

                if (init) {
                    _orientationChangeDefaultAction()
                }
            }
        }

        state: 'Unanimated'
        states: [
            State {
                name: 'Unanimated'
                when: !page.pageContainer || !page._exposed || !completed
                PropertyChanges {
                    target: page
                    restoreEntryValues: false
                    // If a transition is interrupted part way through reset page constants a
                    // transition may have animated to fixed values and not finalised.
                    orientationTransitionRunning: false
                    _orientationPending: false
                }
            },
            State {
                name: 'Portrait'
                when: orientationState.pageOrientation === Qt.PortraitOrientation
                      || orientationState.pageOrientation === Qt.PrimaryOrientation
                         && page.QtQuick.Screen.primaryOrientation === Qt.PortraitOrientation
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: page.QtQuick.Screen.angleBetween(Qt.PortraitOrientation,
                                                               page.QtQuick.Screen.primaryOrientation)
                    orientation: Qt.PortraitOrientation
                }
            },
            State {
                name: 'Landscape'
                when: orientationState.pageOrientation === Qt.LandscapeOrientation
                      || orientationState.pageOrientation === Qt.PrimaryOrientation
                         && page.QtQuick.Screen.primaryOrientation === Qt.LandscapeOrientation
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: page.QtQuick.Screen.angleBetween(Qt.LandscapeOrientation,
                                                               page.QtQuick.Screen.primaryOrientation)
                    orientation: Qt.LandscapeOrientation
                }
            },
            State {
                name: 'PortraitInverted'
                when: orientationState.pageOrientation === Qt.InvertedPortraitOrientation
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: page.QtQuick.Screen.angleBetween(Qt.InvertedPortraitOrientation,
                                                               page.QtQuick.Screen.primaryOrientation)
                    orientation: Qt.InvertedPortraitOrientation
                }
            },
            State {
                name: 'LandscapeInverted'
                when: orientationState.pageOrientation === Qt.InvertedLandscapeOrientation
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: page.QtQuick.Screen.angleBetween(Qt.InvertedLandscapeOrientation,
                                                               page.QtQuick.Screen.primaryOrientation)
                    orientation: Qt.InvertedLandscapeOrientation
                }
            }
        ]

        property Transition defaultTransition: Private.PageOrientationTransition {
            onRunningChanged: orientationState.defaultTransitionRunning = running

            targetPage: page
            orientationChangeActions: [
                ScriptAction {
                    // All orientation-related properties have been changed to their
                    // relevant values, use default action for updating the SafeZoneRect
                    // immediately and changing page geometry with new dimensions.
                    script: page._orientationChangeDefaultAction()
                }
            ]
        }

        Component.onCompleted: {
            if (transitions.length === 0) {
                transitions = [ defaultTransition ]
            }
        }
    }

    Component.onCompleted: {
        _updateIsWindowPageStackContainer()
        orientationState.completed = true
        orientation = orientationState.pageOrientation
    }
}
