/****************************************************************************
**
** SPDX-FileCopyrightText: 2017-2021 Jolla Ltd
** SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Gallery 1.0
import Sailfish.Gallery.private 1.0
import Sailfish.Ambience 1.0
import Sailfish.Share 1.0
import Nemo.FileManager 1.0
import Aurora.Controls 1.0
import Nemo.DBus 2.0

Item {
    id: overlay

    property QtObject player
    property bool active: true
    property bool viewerOnlyMode
    property bool mouseIsPresent

    property alias toolbar: toolbar
    property alias additionalActions: additionalActionsLoader.sourceComponent
    property alias detailsButton: detailsButton
    property bool localFile: fileInfo.localFile
    property alias editingAllowed: editButton.visible
    property alias deletingAllowed: deleteButton.visible
    property alias sharingAllowed: shareButton.visible
    property bool ambienceAllowed: isImage && !error
    property bool printingAllowed: isImage && !error
    readonly property bool allowed: isImage || localFile
    readonly property bool playing: player && player.playing
    property alias topFade: topFade
    property alias fadeOpacity: topFade.fadeOpacity

    property url source
    property string itemId
    property bool isImage
    property bool error
    property int duration: 1
    readonly property int _duration: {
        if (player && player.loaded) {
            return player.duration / 1000
        } else {
            return duration
        }
    }
    property bool isLastItem
    property bool isFirstItem
    readonly property int horizontalPadding: 44

    signal moveForward
    signal moveBack
    signal createPlayer
    signal remove
    signal edited(string image)

    function seekForward() {
        seek(10000, true /*relative*/)
    }

    function seekBackward() {
        seek(-10000, true /*relative*/)
    }

    function triggerAction(action, immediately) {
        if (action === "edit") {
            if (!editPageLoader.active && pageStack.currentPage !== editPageLoader.item) {
                editPageLoader.active = true
                pageStack.animatorPush(editPageLoader.item,
                                       {},
                                       immediately ? PageStackAction.Immediate : PageStackAction.Animated)
            }
        } else if (action === "share") {
            if (player && player.playing) {
                player.pause()
            }
            if (localFile) {
                shareAction.resources = [overlay.source.toString()]
            } else {
                shareAction.resources = [{ "type": "text/x-url", "status": overlay.source.toString() }]
            }
            shareAction.trigger()
        }
    }

    function seek(position, relative) {
        // don't hide controls just yet if user is seeking
        if (hideOverlayTimer.running) {
            hideOverlayTimer.restart()
        }

        if (!player) createPlayer()
        var loaded = player.loaded
        if (!loaded) {
            player.pause() // force load
        }

        player.busy = true

        if (relative) {
            if (!loaded) {
                // cannot jump 10s before knowing the duration of the clip
                delayedRelativeSeek.pending = true
                delayedRelativeSeek.position = position
            } else {
                seekTimer.position = Math.max(Math.min(player.position + position, _duration * 1000), 0)
            }
        } else {
            seekTimer.position = position
        }
        seekTimer.restart()
    }

    // Dummy functions for the AppBarButton
    function _resetContextNow() {}
    function _updateComponentsWidth() {}

    enabled: allowed && source != ""

    ShareAction {
        id: shareAction

        mimeType: localFile ? fileInfo.mimeType : "text/x-url"
    }

    Connections {
        id: delayedRelativeSeek

        property bool pending
        property int position

        target: player

        onLoadedChanged: {
            if (pending) {
                overlay.seek(position, true /* relative */)
                pending = false
            }
        }
    }

    Timer {
        id: seekTimer

        property int position

        interval: 16

        onTriggered: {
            if (player.loaded) {
                player.seek(position)
            } else {
                restart()
            }
        }
    }

    Behavior on opacity { FadeAnimator {}}

    FadeGradient {
        id: topFade

        width: parent.width
        height: detailsButton.height + 2 * detailsButton.y
        topDown: true
    }

    FadeGradient {
        id: bottomFade

        fadeOpacity: topFade.fadeOpacity
        width: parent.width
        height: toolbar.height + 2 * toolbarParent.anchors.bottomMargin
        anchors.bottom: parent.bottom
    }

    AppBarButton {
        id: detailsButton
        objectName: "MoreButton"

        anchors {
            top: parent.top
            topMargin: Theme.paddingLarge + SafeZoneRect.insets.top
            left: parent.left
            leftMargin: Theme.dp(overlay.horizontalPadding)
        }

        visible: overlay.active && overlay.localFile
        icon.source: "image://theme/icon-splus-more"

        onClicked: popupMenu.open(detailsButton)
    }

    AppBarButton {
        id: leftButton

        anchors {
            bottom: toolbarParent.bottom
            left: parent.left
            leftMargin: Theme.dp(overlay.horizontalPadding)
        }

        visible: overlay.mouseIsPresent
        opacity: (overlay.mouseIsPresent && containsMouse && !overlay.active)
                 || (overlay.active && (isImage || toolbar.expanded))
                 ? 1.0
                 : 0.0
        icon.source: "image://theme/icon-splus-left-alt"

        onClicked: moveBack ()

        Rectangle {
            anchors.fill: parent
            color: Theme.rgba("#FFFFFF", 0.2)
            radius: Theme.dp(6)
            visible: overlay.mouseIsPresent && parent.containsMouse && !overlay.active

            Behavior on visible { FadeAnimator {}}
        }
    }

    AppBarButton {
        id: rightButton

        anchors {
            bottom: toolbarParent.bottom
            right: parent.right
            rightMargin: Theme.dp(overlay.horizontalPadding)
        }

        visible: overlay.mouseIsPresent
        opacity: (overlay.mouseIsPresent && containsMouse  && !overlay.active)
                 || (overlay.active && (isImage || toolbar.expanded))
                 ? 1.0
                 : 0.0
        icon.source: "image://theme/icon-splus-right-alt"

        onClicked: moveForward()

        Rectangle {
            anchors.fill: parent
            color: Theme.rgba("#FFFFFF", 0.2)
            radius: Theme.dp(6)
            visible: overlay.mouseIsPresent && parent.containsMouse && !overlay.active

            Behavior on visible { FadeAnimator {}}
        }
    }

    PopupMenu {
        id: popupMenu

        PopupMenuItem {
            objectName: "InfoItem"

            //% "File information"
            text: qsTrId("gallery-me-file_info")
            icon.source: "image://theme/icon-splus-about"

            onClicked: pageStack.animatorPush("DetailsPage.qml", { 'source': overlay.source, 'isImage': overlay.isImage } )
        }

        PopupMenuItem {
            id: ambienceButton
            objectName: "AmbienceItem"

            property bool suppressClick

            //% "Create ambience"
            text: qsTrId("gallery-me-create_ambience")
            visible: isImage
            enabled: ambienceAllowed
            icon.source: "image://theme/icon-splus-ambience"

            onClicked: {
                if (suppressClick) return

                suppressClick = true
                Ambience.create(overlay.source, function(ambienceId) {
                    pageStack.animatorPush("ru.omp.gallery.ambience.AmbienceSettingsDialog", { contentId: ambienceId })
                    ambienceButton.suppressClick = false
                })
            }
        }

        PopupMenuItem {
            objectName: "PrintItem"

            //% "Print"
            text: qsTrId("gallery-me-print")
            visible: isImage
            enabled: printingAllowed
            icon.source: "image://theme/icon-splus-print"

            onClicked: {
                pageStack.animatorPush("Sailfish.Pickers.PrinterPickerPage", { "printPath": fileInfo.file })
            }
        }

        PopupMenuItem {
            //% "Share"
            text: qsTrId("gallery-me-share_file")
            visible: !overlay.error && overlay.localFile && fileInfo.exists && !isImage
            icon.source: "image://theme/icon-splus-share"

            onClicked: overlay.triggerAction("share")
        }

        PopupMenuItem {
            //% "Delete"
            text: qsTrId("gallery-me-delete_file")
            visible: overlay.localFile && fileInfo.exists && !isImage
            icon.source: "image://theme/icon-splus-delete"

            onClicked: overlay.remove()
        }
    }

    Timer {
        id: hideOverlayTimer
        interval: 4000
        running: overlay.active && playing
        onTriggered: {
            if (positionSlider.pressed) {
                restart()
            } else {
                overlay.active = false
            }
        }
    }

    Column {
        id: toolbarParent

        anchors  {
            left: parent.left
            leftMargin: SafeZoneRect.insets.left
            right: parent.right
            rightMargin: SafeZoneRect.insets.right
            bottom: parent.bottom
            bottomMargin: parent.height/20
        }

        height: (isImage ? 0 : sliderRow.height) + (toolbar.expanded ? toolbar.height : 0)

        Behavior on height { NumberAnimation { duration: 200; easing.type: Easing.InOutQuad }}

        Row {
            id: sliderRow

            width: parent.width
            visible: overlay.active && !isImage

            Slider {
                id: positionSlider

                enabled: !overlay.error
                opacity: overlay.error ? 0.0 : 1.0
                rightMargin: Theme.paddingLarge
                width: parent.width - downIcon.width - Theme.horizontalPageMargin

                handleVisible: false
                minimumValue: 0
                maximumValue: 1

                valueText: {
                    var position = overlay._duration * value
                    return Format.formatDuration(position, position >= 3600
                                                 ? Format.DurationLong
                                                 : Format.DurationShort)
                }

                onPressed: {
                    if (!overlay.player) {
                        overlay.createPlayer()
                    }

                    if (!overlay.player.loaded) {
                        overlay.player.pause()
                    }
                }

                // An absolute value will be assigned to an object property and the actual seek delayed.
                onReleased: {
                    var relativePosition = value
                    seek(Qt.binding(function() { return relativePosition * Math.max(0, overlay.player.duration) }))
                }

                Connections {
                    target: player
                    onPositionChanged: {
                        if (!positionSlider.pressed && player.duration > 0 && !seekTimer.running) {
                            positionSlider.value = player.position / player.duration
                        }
                    }
                    onSourceChanged: positionSlider.value = 0
                }

                Timer {
                    interval: 500
                    repeat: true
                    running: player && player.busy && Qt.application.active
                    onRunningChanged: positionSlider.progressOpacity = 1.0
                    onTriggered: positionSlider.progressOpacity = positionSlider.progressOpacity >= 0.99 ? Theme.highlightBackgroundOpacity : 1.0
                }

                property real progressOpacity: 1.0

                Behavior on progressOpacity {
                    NumberAnimation { duration: 400; easing.type: Easing.InOutQuad }
                }

                Binding {
                    target: positionSlider._progressBarItem
                    property: "opacity"
                    value: positionSlider.progressOpacity
                }

            }

            IconButton {
                id: downIcon

                anchors {
                    verticalCenter: parent.verticalCenter
                    verticalCenterOffset: positionSlider._valueLabel.height/4
                }

                visible: !overlay.error || (fileInfo.localFile && fileInfo.exists)
                icon.source: "image://theme/icon-splus-up-alt"
                icon.rotation: toolbar.expanded ? 0 : 180

                onClicked: toolbar.expanded = !toolbar.expanded
            }
        }

        Row {
            id: toolbar

            property bool expanded: isImage && overlay.active

            anchors.horizontalCenter: parent.horizontalCenter
            spacing: isImage ? Theme.paddingLarge : Theme.dp(60)
            opacity: expanded ? 1.0 : 0.0
            enabled: expanded

            Behavior on opacity { FadeAnimator {}}

            Loader {
                id: additionalActionsLoader

                anchors.verticalCenter: parent.verticalCenter
            }

            IconButton {
                anchors.verticalCenter: parent.verticalCenter
                visible: !isImage
                enabled: !overlay.error
                         && (player && player.position !== 0)
                         && overlay.localFile && fileInfo.exists
                opacity: overlay.error ? 0.0 : 1.0
                icon.source: "image://theme/icon-splus-10s-back"

                onClicked: seekBackward()
            }

            IconButton {
                anchors.verticalCenter: parent.verticalCenter
                visible: !isImage
                enabled: !overlay.error
                         && (!player || player.position !== player.duration)
                         && overlay.localFile && fileInfo.exists
                opacity: overlay.error ? 0.0 : 1.0
                icon.source: "image://theme/icon-splus-10s-forward"

                onClicked: seekForward()
            }

            IconButton {
                id: deleteButton

                anchors.verticalCenter: parent.verticalCenter
                visible: overlay.localFile && fileInfo.exists && isImage
                icon.source: "image://theme/icon-splus-delete"

                onClicked: overlay.remove()
            }

            IconButton {
                id: shareButton

                anchors.verticalCenter: parent.verticalCenter
                visible: !overlay.error && overlay.localFile && fileInfo.exists && isImage
                icon.source: "image://theme/icon-splus-share"

                onClicked: overlay.triggerAction("share")
            }

            IconButton {
                id: editButton

                anchors.verticalCenter: parent.verticalCenter
                visible: !overlay.error
                         && isImage && !viewerOnlyMode
                         && ImageWriter.isMimeTypeSupported(fileInfo.mimeType)
                icon.source: "image://theme/icon-splus-edit"

                onClicked: {
                    if (ambienceButton.suppressClick) return
                    overlay.triggerAction("edit")
                }

                Loader {
                    id: editPageLoader

                    active: false
                    sourceComponent: ImageEditDialog {
                        source: overlay.source
                        onEdited: overlay.edited(target)
                        onFinished: editPageLoader.active = false

                        Rectangle {
                            anchors.fill: parent
                            z: 1000
                            color: "black"
                            enabled: editInProgress || editSuccessful
                            opacity: enabled ? 1.0 : 0.0
                            parent: overlay

                            BusyIndicator {
                                size: BusyIndicatorSize.Large
                                anchors.centerIn: parent
                                running: editInProgress
                            }

                            TouchBlocker {
                                anchors.fill: parent
                            }
                        }
                    }
                }
            }
        }
    }

    FileInfo {
        id: fileInfo

        url: overlay.source
    }

    Component {
        id: remorsePopupComponent

        RemorsePopup {}
    }

    DBusInterface {
        bus: DBus.SystemBus
        service: "com.nokia.mce"
        path: "/com/nokia/mce/signal"
        iface: "com.nokia.mce.signal"
        signalsEnabled: true

        function mouse_available_state_ind(value) {
            overlay.mouseIsPresent = value === "available"
        }
    }

    DBusInterface {
        id: mceDBusMousePresentRequest

        bus: DBus.SystemBus
        service: 'com.nokia.mce'
        path: '/com/nokia/mce/request'
        iface: 'com.nokia.mce.request'

        function checkMOuseIsPresent() {
            typedCall('mouse_available_state_req', [], function (result) {
                overlay.mouseIsPresent = result === 'available'
            })
        }
    }

    Component.onCompleted: mceDBusMousePresentRequest.checkMOuseIsPresent()
}
