/* -*- indent-tabs-mode: nil; js-indent-level: 2 -*- */
/* vim: set ts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

const { FileUtils } = ChromeUtils.import("resource://gre/modules/FileUtils.jsm");
const { Downloads } = ChromeUtils.import("resource://gre/modules/Downloads.jsm");

Logger.debug("JSScript: PrintHelper.js loaded");

var gPrintHelper = null;

function debug(aMsg) {
  Logger.debug("PrintHelper.js:", aMsg);
}

function isPrintButton(aElement) {
  return aElement.tagName === "BUTTON" &&
         aElement.id === "print";
}

function PrintDialog(aData) {
  this._init(aData)
}

PrintDialog.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver]),

  _init: function(aData) {
    addMessageListener("embedui:printresponse", this);
    sendAsyncMessage("embed:print", aData);
  },

  receiveMessage: function(aMessage) {
    removeMessageListener("embedui:printresponse", this);
    this.onDone();
  },

  onDone: function(aResult) {}
};

function PrintHelper() {
  this._init();
}

PrintHelper.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver,
                                          Ci.nsISupportsWeakReference]),

  _dialog: null, // a proxy for modal subwindow
  _fileName: "",

  _init: function() {
    addEventListener("click", this, false);
    addMessageListener("embedui:print", this);
  },

  handleEvent: function(aEvent) {
    switch (aEvent.type) {
      case "click":
      this.onClicked(aEvent);
      break;
    }
  },

  receiveMessage: function receiveMessage(aMessage) {
    // We recieve from qml already unique file and correct file
    this._fileName = aMessage.data.to;

    let self = this;
    (async function() {
      let download = await Downloads.createDownload({
        source: Services.ww.activeWindow,
        target: aMessage.data.to,
        saver: "pdf",
        contentType: "application/pdf"
      });
      download.start();
      download.onchange = function() {
        if (download.error) {
          sendAsyncMessage("embed:printpdfdownloaderror", {});
          return;
        }
        if (download.succeeded) {
          self._createPrintDialog();
        }
      }
    })().then(null, Cu.reportError);
  },

  reset: function() {
    this._dialog = null;
  },

  onClicked: function(aEvent) {
    if (!isPrintButton(aEvent.target)) {
      return;
    }
    let randomStr = (Math.random() + 1).toString(36).substring(2);
    let fileName = "/tmp/" + randomStr + ".pdf";
    this._fileName = fileName;

    (async function() {
      let download = await Downloads.createDownload({
        source: content.window.location.href,
        target: fileName,
      });
      download.start();
    })().then(null, Cu.reportError);

    this._createPrintDialog();
  },

  _createPrintDialog: function _createPrintDialog() {
    let that = this;

    this._dialog = new PrintDialog({
      "filePath": this._fileName
    });

    this._dialog.onDone = function (result) {
      that.reset();
    }
  }
};

gPrintHelper = new PrintHelper();
 