/*
 * SPDX-FileCopyrightText: Copyright 2022-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2018 – 2020 Jolla Ltd.
 *
 * License: Proprietary
 */

pragma Singleton

import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import org.nemomobile.configuration 1.0

Item {
    id: root

    property PageStack pageStack

    /*!
    \internal

    Implementation detail for file manager
    */
    property FileManagerNotification errorNotification

    property alias gridLayoutSetting: gridLayoutSetting
    property alias hiddenItemsSetting: hiddenItemsSetting
    property bool openArchiveIsCorrupted
    property bool homePageSelectionMode
    property bool homePageSelectOneItem
    property var multiselectInitialSelectedFiles: []
    property bool multiselectionInProgress

    // Current list of mimeTypes: https://confluence.omprussia.ru/x/2zCbAw
    readonly property var supportedMimeTypesOfImages: [
        "image/bmp",
        "image/gif",
        "image/ico",
        "image/icon",
        "image/jpeg",
        "image/pjpeg",
        "image/png",
        "image/svg+xml",
        "image/tiff",
        "image/vnd.microsoft.icon",
        "image/webp",
        "image/x-MS-bmp",
        "image/x-bmp",
        "image/x-ico",
        "image/x-icon",
    ]

    signal updatePopUpContent
    signal selectedFilesCleared

    // Call before start to use
    function init(pageStack) {
        root.pageStack = pageStack
    }

    function openDirectory(properties) {
        if (!properties.hasOwnProperty("errorNotification")) {
            createErrorNotification()
            properties["errorNotification"] = root.errorNotification
        }

        return pageStack.animatorPush(Qt.resolvedUrl("DirectoryPage.qml"), properties)
    }

    function openGalleryPreviewPage(path, mimeType, extractor) {
        const properties = __generatePreviewProperties(path, mimeType, extractor)

        if (Object.keys(properties).length > 0) {
            return pageStack.animatorPush(Qt.resolvedUrl("GalleryPreviewPage.qml"), properties)
        } else {
            return null
        }
    }

    function __generatePreviewProperties(path, mimeType, extractor) {
        var properties = {}

        if (isPicture(mimeType))
            properties["imagePath"] = path

        if (isVideo(mimeType))
            properties["videoPath"] = path

        if (Object.keys(properties).length > 0)
            properties["extractor"] = extractor

        return properties
    }

    function openArchive(file, path, baseExtractionDirectory, extractor, isOpen, archivePageComponent, rejectedPageName, stackAction) {
        createErrorNotification()
        openArchiveIsCorrupted = false
        stackAction = stackAction || PageStackAction.Animated

        extractor.reset({ force: true })
        extractor.archiveFile = file
        extractor.path = path || "/"
        extractor.isOpen = isOpen
        extractor.destinationPageName = "__archive_page"
        extractor.passwordNeeded = false

        if (rejectedPageName !== undefined) {
            extractor.rejectedPageName = rejectedPageName
        } else {
            extractor.rejectedPageName = "__directory_page"
        }

        var properties = {
            archiveFile: file,
            path: path || "/",
            fileName: extractor.fileName,
            errorNotification: errorNotification,
            extractor: extractor
        }

        if (baseExtractionDirectory) {
            properties["baseExtractionDirectory"] = baseExtractionDirectory
        }

        return pageStack.animatorPush(Qt.resolvedUrl("ArchivePage.qml"), properties, stackAction)
    }

    function createErrorNotification() {
        if (!errorNotification) {
            errorNotification = errorNotificationComponent.createObject(root)
        }
    }

    function pathToUrl(path) {
        if (path.indexOf("file://") == 0) {
            console.warn("pathToUrl() argument already url:", path)
            return path
        }

        return "file://" + path.split("/").map(encodeURIComponent).join("/")
    }

    function urlToPath(path) {
        return FileEngine.urlToPath(path)
    }

    function objectDirectory(path) {
        return FileEngine.objectDirectory(path)
    }

    function isPicture(mimeType) {
        return root.supportedMimeTypesOfImages.indexOf(mimeType) !== -1
    }

    function isVideo(mimeType) {
        return mimeType.indexOf("video") === 0
    }

    function findParentWithProperty(item, propertyName) {
        var parentItem = item ? item.parent : null
        while (parentItem) {
            if (parentItem.hasOwnProperty(propertyName)) {
                return parentItem
            }
            parentItem = parentItem.parent
        }
        return null
    }

    function findPage(item) {
        var r = item.parent
        while (r && !r.hasOwnProperty('__silica_page'))
            r = r.parent
        return r
    }

    Component {
        id: errorNotificationComponent

        FileManagerNotification {}
    }

    ConfigurationValue {
        id: gridLayoutSetting
        key: "/apps/ru.omp.filemanager/gridLayout"
        defaultValue: false
    }

    ConfigurationValue {
        id: hiddenItemsSetting

        key: "/apps/ru.omp.filemanager/show_hidden_items"
        defaultValue: false

        onValueChanged: {
            TrackerEngine.showHidden = hiddenItemsSetting.value
        }
    }
}
