/*
 * SPDX-FileCopyrightText: 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2016 – 2019 Jolla Ltd.
 * Copyright (c) 2019 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Nemo.Thumbnailer 1.0

Page {
    id: root

    property string fileName
    property string mimeType
    property bool isDir
    property date modified
    property double size
    property string path
    property int itemCount

    readonly property int _headersHorizontalAlignment: Qt.AlignLeft

    ListModel {
        id: detailsModel

        readonly property string contentsLabel: {
            du.status !== DiskUsage.Idle
            //: Shown when counting number of items a directory (context menu -> details)
            //% "Counting…"
                    ? qsTrId("filemanager-la-counting")
                      //% "%n items, totalling %1."
                    : qsTrId("filemanager-la-items", root.itemCount).arg(Format.formatFileSize(root.size))
        }

        function updateContentsLabel() {
            for (var i = 0; i < detailsModel.count; ++i) {
                if (detailsModel.get(i).isContentsLabel) {
                    detailsModel.setProperty(i, "value", detailsModel.contentsLabel)
                    return
                }
            }
        }

        function addDetail(label, value, isVisible, isContentsLabel) {
            if (isContentsLabel === undefined)
                isContentsLabel = false

            detailsModel.append({
                                    "label": label,
                                    "value": value,
                                    "isVisible": isVisible,
                                    "isContentsLabel": isContentsLabel
                                })
        }

        dynamicRoles: true
    }

    SilicaFlickable {
        objectName: "DetailsPageSilicaFlickable"

        anchors.fill: parent
        contentHeight: column.height + Theme.paddingMedium

        Column {
            id: column

            readonly property bool isVideo: FileManager.isVideo(root.mimeType)

            width: parent.width
            spacing: Theme.paddingLarge

            PageHeader {
                objectName: "InformationPageHeader"

                //% "Directory information"
                title: isDir ? qsTrId("filemanager-he-dir-info")
                               //% "File information"
                             : qsTrId("filemanager-he-file-info")
            }

            Loader {
                id: previewLoader

                width: parent.width
                sourceComponent: FileManager.isPicture(root.mimeType) || column.isVideo
                                 ? previewPictureComponent
                                 : null
            }

            Component {
                id: previewPictureComponent

                Thumbnail {
                    id: previewPicture

                    readonly property int maxHeight: Theme.itemSizeHuge + Theme.itemSizeSmall

                    objectName: "PreviewPictureThumbnail"

                    anchors.horizontalCenter: parent.horizontalCenter
                    width: parent.width - 2 * Theme.horizontalPageMargin
                    height: maxHeight

                    sourceSize {
                        width: previewPicture.width
                        height: previewPicture.height
                    }

                    mimeType: root.mimeType
                    priority: Thumbnail.HighPriority
                    fillMode: Thumbnail.PreserveAspectFit
                    source: path

                    BusyIndicator {
                        id: thumbnailBusyIndicator

                        objectName: "ThumbnailBusyIndicator"

                        anchors.centerIn: parent
                        z: parent.z + 1
                        running: previewPicture.status === Thumbnail.Loading
                        visible: running
                        size: BusyIndicatorSize.Medium

                        FadeAnimator {
                            target: previewPicture
                            easing.type: Easing.InExpo
                            to: 1.0
                            running: !thumbnailBusyIndicator.running
                        }
                    }

                    // For videos that don't have a preview
                    Rectangle {
                        anchors.horizontalCenter: parent.horizontalCenter
                        width: Theme.itemSizeExtraLarge * 4 - 2 * Theme.horizontalPageMargin
                        height: previewPicture.maxHeight
                        visible: previewPicture.status === Thumbnail.Error
                        color: "black"
                    }
                }
            }

            // TODO: waiting for the designers to draw new icons
            Component {
                id: detailIconComponent

                DetailsIcon {
                    objectName: "DetailIcon"

                    anchors.horizontalCenter: parent.horizontalCenter
                    width: parent.width - 2 * Theme.horizontalPageMargin
                    backgroundWidth: Theme.iconSizeExtraLarge
                    backgroundHeight: backgroundWidth
                    backgroundRadius: Theme.paddingMedium
                    iconSource: Theme.iconForMimeType(root.mimeType)
                }
            }

            Repeater {
                model: detailsModel
                delegate: DetailItem {
                    objectName: "FileDetailItem"

                    label: model.label
                    value: model.value
                    visible: model.isVisible
                    forceValueBelow: true
                    alignment: Qt.AlignLeft
                    _valueItem.font.pixelSize: Theme.fontSizeMedium
                    _valueItem.color: Theme.primaryColor
                }
            }
        }

        VerticalScrollDecorator {}
    }

    DiskUsage { id: du }

    Component.onCompleted: {
        if (isDir) {
            du.calculate(path, function (usage) {
                size = usage[path]
                detailsModel.updateContentsLabel()
            })

            du.fileCount(path, function(count) {
                itemCount = count
                detailsModel.updateContentsLabel()
            }, DiskUsage.Files | DiskUsage.Dirs)
        }

        //% "Name"
        detailsModel.addDetail(qsTrId("filemanager-he-name"),
                               root.fileName,
                               true)
        //% "Path to %1"
        detailsModel.addDetail(qsTrId("filemanager-he-path").arg(root.isDir
                                                                   //% "folder"
                                                                 ? qsTrId("filemanager-he-path_folder")
                                                                   //% "file"
                                                                 : qsTrId("filemanager-he-path_file")),
                               root.path,
                               true)
        //% "Size"
        detailsModel.addDetail(qsTrId("filemanager-he-size"),
                               Format.formatFileSize(root.size),
                               !root.isDir)
        //% "Contents"
        detailsModel.addDetail(qsTrId("filemanager-he-contents"),
                               detailsModel.contentsLabel,
                               root.isDir,
                               true)
        //% "Type"
        detailsModel.addDetail(qsTrId("filemanager-he-type"),
                               root.mimeType,
                               true)
        //% "Modified"
        detailsModel.addDetail(qsTrId("filemanager-he-modified"),
                               Format.formatDate(root.modified, Formatter.DateLong),
                               true)
    }
}
