/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Aurora.Controls 1.0
import Aurora.Controls.private 1.0

Popup {
    id: popupRoot

    default property alias items: itemsView.stackChildren
    property alias headerText: itemsView.headerText
    property alias headerBackground: itemsView.headerBackground
    property real _collapsedHeight: Theme.dp(72) * 5 //minimum height of popup when it has handle

    property alias _pagesContainer: pagesContainer

    property var _pages: [ itemsView ]
    property Item _page
    property var _currentPage: itemsView
    property real _contentHeight: _currentPage.pageHeight
    property alias _inTransition: changePageAnimation.inTransition
    property real _toolbarHeight

    function _pushPage(popUpPage) {
        if (_inTransition) {
            return
        }

        changePageAnimation.pushPage(popUpPage)
    }

    function _popPage(popUpPage) {
        if (_inTransition) {
            return
        }

        changePageAnimation.popPage()
    }

    onAboutToShow: heightBehavior.enabled = false

    onClosed: {
        for(var i = 1; i < popupRoot._pages.length; i++) {
            popupRoot._pages[i].visible = false
            popupRoot._pages[i].x = pagesContainer.width
        }
        itemsView.x = 0
        itemsView.visible = true
        itemsView.width = Qt.binding(function() { return pagesContainer.width })
        itemsView.height = Qt.binding(function() { return pagesContainer.height })
        popupRoot._pages = [ itemsView ]
        _currentPage = itemsView
    }

    contentHeight: _contentHeight + Theme.paddingSmall

    Behavior on contentHeight {
        id: heightBehavior

        enabled: false

        NumberAnimation {
            duration: 200
            easing.type: Easing.InOutQuart
        }
    }

    Item {
        id: pagesContainer

        width: parent.width
        height: parent.height - _toolbarHeight

        PopupPage {
            id: itemsView

            width: pagesContainer.width
            height: pagesContainer.height

            _parentPopup: popupRoot
        }

        SequentialAnimation {
            id: changePageAnimation

            property alias rightPage: rightPageAnimator.target
            property alias leftPage: leftPageAnimator.target
            property bool inTransition
            property real duration: 200

            property bool _pushPage: true //pop page if false

            function pushPage(page) {
                heightBehavior.enabled = true
                changePageAnimation._pushPage = true
                changePageAnimation.leftPage = popupRoot._pages[popupRoot._pages.length - 1]
                changePageAnimation.rightPage = page
                changePageAnimation.restart()
            }

            function popPage() {
                heightBehavior.enabled = true
                changePageAnimation._pushPage = false
                changePageAnimation.leftPage = popupRoot._pages[popupRoot._pages.length - 2]
                changePageAnimation.rightPage = popupRoot._pages[popupRoot._pages.length - 1]
                changePageAnimation.restart()
            }

            ScriptAction {
                script: {
                    if (changePageAnimation._pushPage) {
                        if (!changePageAnimation.rightPage || changePageAnimation.leftPage != popupRoot._pages[popupRoot._pages.length - 1]) {
                            console.error("Can't push null!")
                            popupRoot.close()
                            changePageAnimation.stop()
                            return
                        }
                        popupRoot._pages.push(changePageAnimation.rightPage)
                        popupRoot._currentPage = changePageAnimation.rightPage
                        changePageAnimation.rightPage.visible = true
                        changePageAnimation.rightPage.width = Qt.binding(function() { return pagesContainer.width })
                        changePageAnimation.rightPage.height = Qt.binding(function() { return pagesContainer.height })
                    } else {
                        if (popupRoot._pages.length < 2) {
                            console.error("Can't pop page, page count is less than 2")
                            popupRoot.close()
                            changePageAnimation.stop()
                            return
                        }
                        popupRoot._pages.pop()
                        changePageAnimation.leftPage.visible = true
                        popupRoot._currentPage = changePageAnimation.leftPage
                        changePageAnimation.leftPage.width = Qt.binding(function() { return pagesContainer.width })
                        changePageAnimation.leftPage.height = Qt.binding(function() { return pagesContainer.height })
                        changePageAnimation.leftPage._gotoBegin()
                    }
                    changePageAnimation.rightPage._inTransition = true
                    changePageAnimation.leftPage._inTransition = true
                    changePageAnimation.inTransition = true
                }
            }

            ParallelAnimation {
                NumberAnimation {
                    id: leftPageAnimator

                    duration: changePageAnimation.duration
                    easing.type: Easing.InOutQuart
                    property: "x"
                    from: changePageAnimation._pushPage ? 0 : -pagesContainer.width
                    to: changePageAnimation._pushPage ? -pagesContainer.width : 0
                }

                NumberAnimation {
                    id: rightPageAnimator

                    duration: changePageAnimation.duration
                    easing.type: Easing.InOutQuart
                    property: "x"
                    from: changePageAnimation._pushPage ? pagesContainer.width : 0
                    to: changePageAnimation._pushPage ? 0 : pagesContainer.width
                }
            }

            ScriptAction {
                script: {
                    heightBehavior.enabled = false
                    if (changePageAnimation._pushPage) {
                        changePageAnimation.leftPage.visible = false
                        changePageAnimation.rightPage.height = pagesContainer.height // unbind
                        changePageAnimation.rightPage.width = pagesContainer.width // unbind
                    } else {
                        changePageAnimation.rightPage.visible = false
                        changePageAnimation.leftPage.width = pagesContainer.width // unbind
                        changePageAnimation.leftPage.height = pagesContainer.height // unbind
                    }
                    changePageAnimation.rightPage._inTransition = false
                    changePageAnimation.leftPage._inTransition = false
                    changePageAnimation.leftPage = null
                    changePageAnimation.rightPage = null
                    changePageAnimation.inTransition = false
                }
            }
        }
    }

    // we need preferredHeight in _collapsedHeight, but without binding
    Component.onCompleted: _collapsedHeight = popupRoot.preferredHeight
}
