/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.6
import QtQuick.Layouts 1.1
import QtGraphicalEffects 1.0
import Sailfish.Silica 1.0
import Aurora.Controls 1.0
import Aurora.Lottie 1.0

Item {
    id: root

    readonly property bool isLandscape: __silica_applicationwindow_instance.orientation &
                                        Orientation.LandscapeMask

    readonly property bool largeScreen: Screen.sizeCategory >= Screen.Large

    property int orientation: root.isLandscape ? Qt.Horizontal : Qt.Vertical

    property real portraitAnimationSize: Theme.dp(420)
    property real landscapeAnimationSize: root.largeScreen ? portraitAnimationSize : Theme.dp(320)
    property real animationSize: root.portraitAnimationSize
    property bool autoStartOnVisible: true

    property real betweenSectionSpacing: Theme.dp(48)
    property real insideSectionSpacing: Theme.dp(36)

    /*! Play animation on device orientation changes */
    property bool autoStartOnIsLandscapeChanged: true

    /*! Play animation only if layout orientation changed */
    property bool autoStartOnOrientationChange: false

    /*! User defined additional item. */
    property Item additionalContentItem: null

    property alias lottie: lottie
    property alias label: label

    anchors {
        fill: parent
    }

    implicitWidth: rootLayout.implicitWidth
    implicitHeight: rootLayout.implicitHeight

    onIsLandscapeChanged: {
        if (root.autoStartOnIsLandscapeChanged) {
            lottie.start()
        }
    }

    onOrientationChanged: {
        if (root.autoStartOnOrientationChange) {
            lottie.start()
        }
    }

    onVisibleChanged: {
        if (root.visible && root.autoStartOnVisible) {
            lottie.start()
        }
    }

    GridLayout {
        id: rootLayout

        anchors.centerIn: parent

        rowSpacing: root.betweenSectionSpacing
        columnSpacing: rowSpacing

        rows: 2
        columns: 1

        Item {
            Layout.preferredWidth: root.animationSize
            Layout.preferredHeight: root.animationSize
            Layout.alignment: Qt.AlignHCenter

            LottieAnimation {
                id: lottie

                anchors.fill: parent

                loops: 0
                autoPlay: root.autoStartOnVisible

                source: root.animationSource

                layer {
                    enabled: true
                    effect: ColorOverlay {
                        color: Theme.highlightColor
                    }
                }
            }
        }

        ColumnLayout {
            Layout.alignment: Qt.AlignHCenter

            spacing: 0

            Label {
                id: label

                Layout.preferredWidth: Math.min(implicitWidth + Theme.paddingMedium,
                                                root.width / rootLayout.columns)
                Layout.alignment: Qt.AlignHCenter

                horizontalAlignment: Text.AlignHCenter
                elide: Text.ElideRight
                wrapMode: Text.Wrap
                maximumLineCount: 2
                font.pixelSize: Theme.fontSizeExtraLarge
            }

            Item {
                Layout.preferredHeight: root.insideSectionSpacing
                visible: root.additionalContentItem
            }

            Item {
                id: additionalContentItemContainer

                Layout.preferredWidth: additionalContentItem
                                       ? additionalContentItem.implicitWidth
                                       : 0
                Layout.preferredHeight: additionalContentItem
                                        ? additionalContentItem.implicitHeight
                                        : 0
                Layout.alignment: Qt.AlignHCenter

                data: root.additionalContentItem
            }
        }
    }

    states: [
        State {
            when: root.orientation === Qt.Horizontal

            PropertyChanges {
                target: rootLayout

                rows: 1
                columns: 2
            }

            PropertyChanges {
                target: root

                animationSize: root.landscapeAnimationSize
            }
        }
    ]
}
