/*
 * SPDX-FileCopyrightText: Copyright 2019-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2018 – 2019 Jolla Ltd.
 *
 * License: Proprietary
 */
import QtQuick 2.5
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0

Page {
    id: page

    property var path
    property var archiveFile
    property var fileName
    property var baseExtractionDirectory
    property int __archive_page
    readonly property string __directory_path: extractor.isInternalMode ? archiveFile + (path !== "/" ? path : "")
                                                                        : "/" + fileName + (path !== "/" ? path : "")

    property string title
    property FileManagerNotification errorNotification
    property ExtractorView extractor

    signal archiveExtracted(string containingFolder)

    showNavigationIndicator: backNavigation

    Component.onCompleted: {
        if (!errorNotification) {
            errorNotification = errorNotificationComponent.createObject(page)
        }

        FileManager.init(pageStack)
    }

    onStatusChanged: {
        if (status == PageStatus.Activating) {
            extractor.path = page.path
            files.opacity = 1.0
            pathBar.updatePosition()
            archivePathModel.path = page.__directory_path.length > 0 ? page.__directory_path : ""
        }

        if (status == PageStatus.Deactivating) {
            files.opacity = 0.0
        }
    }

    FileGridHeader {
        id: appBar

        headerTitle: page.title.length > 0 ? page.title
                                           : path !== "/" ? extractor.model.currentDirName : fileName
        parentGridView: files
        path: extractor.model.path
        countFiles: extractor.model.count
        errorType: extractor.model.errorState
        isArchiveMode: true
        z: page.z + 1

        onExtractArchive: {
            extractor.rejectedPageName = "__archive_page"
            extractor.selectedExtractPath(extractor.extractAllFiles)
        }
    }

    PathBar {
        id: pathBar

        anchors {
            top: appBar.bottom
            topMargin: Theme.paddingMedium
        }

        z: appBar.z
        model: archivePathModel
        enabled: !busyIndicator.running
        visible: enabled
        homeButtonVisible: extractor.isInternalMode
        homeButtonIcon.source: homeButtonVisible ? "image://theme/icon-splus-home"
                                                 : "image://theme/icon-splus-extract-out"

    }

    DirectoryPathModel {
        id: archivePathModel

        archiveMode: true
    }

    SilicaListView {
        id: files

        readonly property bool archiveMode: true

        objectName: "ArchivePageSilicaListView"
        anchors {
            fill: parent
            topMargin: appBar.height + pathBar.height
        }
        clip: true
        delegate: InternalFileItem {

            function cleanUp() {
                //% "Deleted extracted directory"
                var text = model.isDir ? qsTrId("filemanager-la-deleted_extracted_dir")
                                       : //% "Deleted extracted file"
                                         qsTrId("filemanager-la-deleted_extracted_file")
                remorseAction(text, function() {
                    FileEngine.deleteFiles(model.extractedTargetPath, true)
                    extractor.model.cleanExtractedEntry(model.fileName)
                })
            }

            width: ListView.view.width
            compressed: !model.extracted
            menu: contextMenu
            onClicked: {
                if (model.extracted) {
                    if (model.isDir) {
                        var directory = FileManager.openDirectory({
                                                                      path: model.extractedTargetPath,
                                                                      initialPath: StandardPaths.home,
                                                                      showDeleteFolder: true,
                                                                      //% "Extracted folder"
                                                                      description: qsTrId("filemanager-he-extracted_folder")
                                                                  })
                        directory.folderDeleted.connect(function () {
                            extractor.model.cleanExtractedEntry(model.fileName)
                        })
                    } else {
                        Qt.openUrlExternally(FileManager.pathToUrl(model.extractedTargetPath))
                    }
                } else if (model.isDir) {
                    var obj = FileManager.openArchive(archiveFile, extractor.appendPath(model.fileName), baseExtractionDirectory, extractor, false)
                    obj.pageCompleted.connect(function(archivePage) {
                        archivePage.archiveExtracted.connect(page.archiveExtracted)
                    })
                } else {
                    openMenu()
                }
            }

            Component {
                id: contextMenu
                ContextMenu {
                    ExtractContextMenuItem {
                        objectName: "ExtractHereMenuItem"

                        extractorView: extractor
                        visible: !model.extracted && extractor.extractHereAvailable
                        destinationPageName: "__archive_page"
                        rejectedPageName: destinationPageName
                    }

                    ExtractContextMenuItem {
                        objectName: "ExtractToMenuItem"

                        extractorView: extractor
                        needSelectDirectory: true
                        visible: !model.extracted
                        destinationPageName: "__archive_page"
                        rejectedPageName: destinationPageName
                    }

                    MenuItem {
                        objectName: "DeleteExtractedDirectoryMenuItem"

                        visible: model.extracted
                        //% "Delete extracted directory"
                        text: model.isDir ? qsTrId("filemanager-me-delete_extracted_dir")
                                            //% "Remove extracted file"
                                          : qsTrId("filemanager-me-delete_extracted_file")
                        onClicked: cleanUp()
                    }
                }
            }
        }

        model: extractor.model

        Behavior on opacity { FadeAnimator { duration: 100 } }

        VerticalScrollDecorator {}

        Component {
            id: errorNotificationComponent

            FileManagerNotification {}
        }

        PageBusyIndicator {
            id: busyIndicator

            running: extractor.model.count === 0 && !FileManager.openArchiveIsCorrupted

            z: 1
            FadeAnimator {
                target: busyIndicator
                duration: 1000
                easing.type: Easing.InExpo
                to: 1.0
            }
        }

        Timer {
            id: waitTimer

            running: true
            interval: 15 * 1000
            onTriggered: {
                busyIndicator.running = false
            }
        }
    }

    NoFilesPagePlaceholder {
        objectName: "NoFilesPlaceholder"
        visible: extractor.model.count === 0 && !busyIndicator.running

        label.text: !FileManager.openArchiveIsCorrupted
                      //% "No files"
                    ? qsTrId("filemanager-la-no_files")
                    : qsTrId("filemanager-la-opening_archive_failed")
    }
}
