/****************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/

import QtQuick 2.5
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private

PickerDialog {
    id: root

    property var _contentModel
    property string _searchPlaceholderText
    property alias _emptyPlaceholderText: gridView.emptyPlaceholderText

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: _background ? gridView : __silica_applicationwindow_instance.contentItem
        targetPage: root
    }

    PickerGridView {
        id: gridView

        anchors.topMargin: searchItem.height + searchItem.y + Theme.paddingMedium
        anchors.bottomMargin: toolBar.height + SafeZoneRect.insets.bottom

        isDialog: true
        itemModel: root._contentModel
        model: root._contentModel.model
        selectedCount: _selectedCount
    }

    SearchSlidingItem {
        id: searchItem

        contentModel: _contentModel
        appBarSearchField: searchField
        gridView: gridView
        placeholderText: root._searchPlaceholderText
    }

    PickerDialogHeader {
        id: dialogHeader

        selectedCount: gridView.selectedCount
        divider: searchItem.y === SafeZoneRect.insets.top

        extraContent.width: parent.width
        extraContent.x: 0
        reserveExtraContent: false
        singleSelectionMode: _contentModel.singleSelectionMode

        extraContent.children: Item {
            function _updateComponentsWidth() {} // Mock appbar internal stuff

            anchors.verticalCenter: parent.verticalCenter
            height: parent.height
            width: parent.width

            AppBarSearchField {
                id: searchField

                onTextChanged: {
                    if (visible && _contentModel) {
                        _contentModel.filter = text.toLowerCase().trim()
                        searchItem.text = text
                    }
                }

                onVisibleChanged: {
                    searchItem.enableAnimation = false
                    if (visible) {
                        searchItem.visible = false
                    } else {
                        if (activeFocus) {
                            searchItem.forceActiveFocus()
                            searchItem.active = true
                        }
                        if (text != "") {
                            searchItem.active = true
                        }

                        searchItem.visible = true
                    }

                    searchItem.enableAnimation = true
                }

                EnterKey.onClicked: focus = false

                anchors.centerIn: parent

                visible: (SafeZoneRect._orientation & Orientation.LandscapeMask)
                         && dialogHeader.width > (4 * Theme.paddingLarge
                                                  + dialogTextMetrics.width
                                                  + screen.height * 0.333)
                placeholderText: root._searchPlaceholderText
                backgroundEnabled: true
                preferredWidth: Math.max(screen.height * 0.333, minimalWidth)

                EnterKey.iconSource: "image://theme/icon-m-keyboard-enter-close"
            }
        }

        TextMetrics {
            id: dialogTextMetrics

            text: dialogHeader.cancelText + dialogHeader.acceptText
            font {
                pixelSize: Theme.fontSizeSmall
                family: Theme.fontFamily
            }
        }
    }

    AppBar {
        id: toolBar

        readonly property bool longVersion: {
            return width > (2 * Theme.paddingLarge
                            + Theme.dp(128) + Theme.dp(72 * 3)
                            + 1.5 * Theme.horizontalPageMargin
                            + appBarTextMetrics.width
                            + (searchField.visible ? 0: Theme.dp(64 + 72) + 0.5 * Theme.horizontalPageMargin))
        }

        // Hide toolBar when keyboard is shown
        height: Math.max(Theme._itemSizeHeader - __silica_applicationwindow_instance.pageStack.panelSize, 0)
        _bottomPosition: true
        background: true
        divider: false

        AppBarSpacer {
            visible: toolBar.longVersion
        }

        AppBarButton {
            onClicked: {
                if (searchItem.active) {
                    searchItem.text = ""
                    searchItem.focus = false
                    searchItem.active = false
                } else {
                    searchItem.active = true
                    searchItem.forceActiveFocus()
                }
            }

            icon.source: "image://theme/icon-splus-search"
            //% "Search"
            text: qsTrId("components_pickers-la-search")
            highlighted: searchItem.active
            visible: !searchField.visible
        }

        AppBarSpacer {
            fixedWidth: toolBar.longVersion ? Theme.dp(64) : 0
        }

        AppBarButton {
            onClicked: gridView.gridLayout = !gridView.gridLayout

            icon.source: "image://theme/icon-splus-%1".arg(gridView.gridLayout ? "list" : "cage")

            text: toolBar.longVersion ? (gridView.gridLayout
                                           //% "Show list"
                                         ? qsTrId("components_pickers-la-show_grid")
                                           //% "Show grid"
                                         : qsTrId("components_pickers-la-show_list"))
                                      : ""
        }

        AppBarSpacer {
            visible: toolBar.longVersion
            fixedWidth: Theme.dp(64)
        }


        AppBarButton {
            onClicked: {
                for (var i = 0; i < _contentModel.model.count; i++) {
                    if (_contentModel.model.get(i).selected) {
                        _contentModel.updateSelected(i, false)
                    }
                }
            }
            //% "Clear selection"
            text: toolBar.longVersion ? qsTrId("pickers-me-clear_selection") : ""
            icon.source: "image://theme/icon-splus-multiclean"
        }

        AppBarSpacer {
            visible: toolBar.longVersion
            fixedWidth: Theme.dp(64)
        }

        AppBarButton {
            onClicked: {
                for (var i = 0; i < _contentModel.model.count; i++) {
                    if (!_contentModel.model.get(i).selected) {
                        _contentModel.updateSelected(i, true)
                    }
                }
            }

            //% "Select all"
            text: toolBar.longVersion ? qsTrId("pickers-me-select_all") : ""
            icon.source: "image://theme/icon-splus-multiselect"
        }

        AppBarSpacer {
            visible: toolBar.longVersion
        }
    }

    TextMetrics {
        id: appBarTextMetrics

        //% "Clear selection"
        text: qsTrId("pickers-me-clear_selection")
                //% "Select all"
              + qsTrId("pickers-me-select_all")
                                           //% "Search"
              + (searchField.visible ? "" : qsTrId("components_pickers-la-search"))
              + (gridView.gridLayout
                   //% "Show list"
                 ? qsTrId("components_pickers-la-show_grid")
                   //% "Show grid"
                 : qsTrId("components_pickers-la-show_list"))
        font {
            pixelSize: Theme.fontSizeSmall
            family: Theme.fontFamily
        }
    }
}
