/* -*- indent-tabs-mode: nil; js-indent-level: 2 -*- */
/* vim: set ts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

Logger.debug("JSScript: DatetimePicker.js loaded");

var gDatetimeHelper = null;

function debug(aMsg) {
  Logger.debug("DateTimePicker.js:", aMsg);
}

function isDateTimeInput(aElement) {
  if(aElement.tagName === "INPUT"){
      let validType = ['date', 'datetime-local', 'time'];
      return validType.includes(aElement.type);
  }
  return false;
}

function DateDialog(aData) {
  this._init(aData)
}

DateDialog.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver]),

  _init: function(aData) {
    addMessageListener("embedui:datetimeresponse", this);
    sendAsyncMessage("embed:datetime", aData);
  },

  receiveMessage: function(aMessage) {
    removeMessageListener("embedui:datetimeresponse", this);
    this.onDone(aMessage.data);
  },

  onDone: function(aResult) {}
};

function DatetimeHelper() {
  this._init();
}

DatetimeHelper.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver,
                                          Ci.nsISupportsWeakReference]),

  _inputElement: null,
  _dialog: null, // a proxy for modal subwindow

  _init: function() {
    addEventListener("click", this, false);
  },

  handleEvent: function(aEvent) {
    switch (aEvent.type) {
      case "click":
      this.onClicked(aEvent);
      break;
    }
  },

  reset: function() {
    this._inputElement.blur();
    this._inputElement = null;
    this._dialog = null;
  },

  onClicked: function(aEvent) {
    let target = aEvent.target.shadowRoot ? aEvent.target.shadowRoot.activeElement
                                          : aEvent.target
    // don't use 'this' inside lambdas for the sake of clarity
    let that = this;

    if (this._inputElement === null && isDateTimeInput(target)) {
      this._inputElement = target;
      var data = {"type": target.type}
      var currentdate = new Date();
      if (target.value) {
        var datetime = target.value.split(/-|T|:/);
        if (datetime.length >= 3) {
          currentdate.setFullYear(datetime[0]);
          currentdate.setMonth(datetime[1] - 1);
          currentdate.setDate(datetime[2]);
        } else {
          currentdate.setHours(datetime[0]);
          currentdate.setMinutes(datetime[1]);
        }

        if (datetime.length >= 5) {
          currentdate.setHours(datetime[3]);
          currentdate.setMinutes(datetime[4]);
        }
      }

      data["date"] = currentdate;

      this._dialog = new DateDialog(data);

      this._dialog.onDone = function (result) {

        if (!result) {
          debug("Empty result");
          that.reset();
          return;
        }

        that.onResultsReceived(target, result);
        that.reset();
      }
    } else if (this._inputElement && this._inputElement !== target) {
      // in case user clicked outside of subwindow
      debug("User manage to click outside of subwindow");
      this._dialog.abort();
      this.reset();
    }
  },

  onResultsReceived: function (aElement, aValue) {
    switch (this._inputElement.type) {
      case "time":
        this._inputElement.value = ("0" + aValue.hour).slice(-2)
                                 + ":" + ("0" + aValue.minute).slice(-2);
        break;
      case "date":
        this._inputElement.value = aValue.year
                                 + "-" + ("0" + aValue.month).slice(-2)
                                 + "-" + ("0" + aValue.day).slice(-2);
        break;
      case "datetime-local":
        this._inputElement.value = aValue.year
                                 + "-" + ("0" + aValue.month).slice(-2)
                                 + "-" + ("0" + aValue.day).slice(-2)
                                 + "T" + ("0" + aValue.hour).slice(-2)
                                 + ":" + ("0" + aValue.minute).slice(-2);
      default:
        break;
    }

    this.fireOnChange(aElement);
  },

  fireOnChange: function(aElement) {
    let evt = aElement.ownerDocument.createEvent("Events");
    evt.initEvent("change", true, true, aElement.defaultView, 0,
                  false, false,
                  false, false, null);
    content.window.setTimeout(function() {
        aElement.dispatchEvent(evt);
    }, 0);
  }
};

gDatetimeHelper = new DatetimeHelper();
