/****************************************************************************************
**
** SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Aurora.Controls 1.0

Item {
    id: page

    default property alias stackChildren: stackColumn.children
    property alias headerText: header.text
    property alias headerBackground: header._headerBackground
    property real pageHeight
    readonly property real contentHeight: _private.fullContentHeight
    property Popup _parentPopup
    property bool _isSubMenu
    property bool _inTransition

    function _defaultState() {
        if (!_private.handleEnabled) {
            flickable.contentY = 0
            return
        }

        // set values and binding, that equal to "minHeight" state
        pageHeight = Qt.binding(function () {
            return Math.max(Math.min(flickable.contentY + _private.calcMinHeight, _private.calcMaxHeight), _private.calcMinHeight)
        })
        stackColumn.y = Qt.binding(function () {
           return Math.max(flickable.height - page.pageHeight, 0) + header.height
        })
        flickable.contentY = 0
    }

    function _gotoBegin() {
        if (!_private.handleEnabled || stackColumn.y < 0) {
            return
        }
        flickable.contentY = stackColumn.y - header.height
    }

    onStackChildrenChanged: {
        if (stackChildren[stackChildren.length - 1] && stackChildren[stackChildren.length - 1].hasOwnProperty("_parentPopup")) {
            stackChildren[stackChildren.length - 1]._parentPopup = _parentPopup
        }
    }

    on_ParentPopupChanged: {
        for (var i = 0; i < stackChildren.length; i++) {
            if (stackChildren[i].hasOwnProperty("_parentPopup")) {
                stackChildren[i]._parentPopup = _parentPopup
            }
        }
    }

    states: [
        State {
            name: ""
            PropertyChanges {
                target: page
                pageHeight: _private.calcMaxHeight
            }
        },
        State {
            name: "invisible"
            when: !visible
        },
        State {
            name: "withoutHandle"
            when: visible && !_private.handleEnabled
            PropertyChanges {
                target: page
                pageHeight: Math.min(contentHeight, _private.maxHeight)
            }
            PropertyChanges {
                target: flickable
                height: pageHeight
                y: 0
            }
        },
        State {
            name: "lessMaxHeight"
            when: visible && _private.handleEnabled && flickable.contentY >= 0 && (flickable.contentY < _private.calcMaxHeight - _private.calcMinHeight)
            PropertyChanges {
                target: page
                pageHeight: Math.max(Math.min(flickable.contentY + _private.calcMinHeight, _private.calcMaxHeight), _private.calcMinHeight)
            }
        },
        State {
            name: "maxHeight"
            when: visible && _private.handleEnabled && (flickable.contentY >= _private.calcMaxHeight - _private.calcMinHeight)
            PropertyChanges {
                target: page
                pageHeight: _private.calcMaxHeight
            }
        },
        State {
            name: "closed"
            when: visible && _parentPopup && !_parentPopup.opened
        }
    ]

    QtObject {
        id: _private

        readonly property real minHeight:  _parentPopup ? _parentPopup._collapsedHeight : Theme.dp(75) * 5
        readonly property real maxHeight: _parentPopup ? _parentPopup.implicitHeight : Theme.dp(72) * 9
        readonly property real toolbarHeight: _parentPopup ? _parentPopup._toolbarHeight : 0
        readonly property real fullContentHeight: stackColumn.height + Math.max(header.height, Theme.paddingSmall) + _private.toolbarHeight + Theme.paddingSmall
        readonly property real calcMinHeight: Math.min(_private.minHeight, _private.fullContentHeight)
        readonly property real calcMaxHeight: Math.min(_private.maxHeight, _private.fullContentHeight)
        readonly property real calcHeightDiff: Math.max(0, calcMaxHeight - calcMinHeight)
        readonly property bool handleEnabled: {
            if (_parentPopup) {
                switch(_parentPopup.contentItem.transformOrigin) {
                case Item.Bottom:
                case Item.BottomLeft:
                case Item.BottomRight:
                    return _parentPopup ? fullContentHeight > _parentPopup.preferredHeight : true
                }
            }
            return false
        }

        onHandleEnabledChanged: flickable.correctHeight(false,false)
    }

    Connections {
        onAboutToHide: flickable.interactive = false
        onAboutToShow: flickable.correctHeight(true, false)
        onOpened: flickable.interactive = true
        onClosed: _defaultState()

        target: _parentPopup
    }

    SilicaFlickable {
        id: flickable

        readonly property real allContentHeight: Math.max(header.height, Theme.paddingSmall) + stackColumn.height + _private.toolbarHeight

        function correctHeight(bindHeight, bindY) {
            if (!_private.handleEnabled) {
                return
            }

            if (bindHeight) {
                flickable.height = Qt.binding(function() {return page.pageHeight})
            } else {
                flickable.height = page.pageHeight
            }

            if (bindY) {
                flickable.y = Qt.binding(function() {
                    return page.pageHeight - flickable.height
                })
            } else {
                flickable.y = 0
            }
        }

        onDragStarted: flickable.correctHeight(false, true)
        onDragEnded: flickable.correctHeight(false, false)
        onFlickStarted: flickable.correctHeight(true, false)
        onFlickEnded: flickable.correctHeight(false, false)

        y: _private.handleEnabled ? pageHeight - _private.calcMinHeight : 0
        width: parent.width
        height: _private.handleEnabled ? _private.calcMinHeight : _private.calcMaxHeight
        quickScrollEnabled: false
        contentHeight: {
            var result = flickable.allContentHeight
            if (_private.handleEnabled) {
                if (pageHeight == _private.calcMaxHeight) {
                    result += _private.calcHeightDiff + Theme.paddingSmall
                } else {
                    result += pageHeight - _private.calcMinHeight + Theme.paddingSmall
                }
            }
            return result
        }

        Column {
            id: stackColumn

            anchors{
                left: parent.left
                right: parent.right
            }

            states: [
                State {
                    name: ""
                    PropertyChanges {
                        target: stackColumn
                        y: {
                            if (page.pageHeight < _private.calcMaxHeight && page.pageHeight >= flickable.height) {
                                return Math.max(flickable.contentY, 0) + header.height
                            }
                            return header.height
                        }
                    }
                    PropertyChanges {
                        target: scrollDecorator
                        y: scrollDecorator.headerHeight
                    }
                },
                State {
                    name: "invisible"
                    when: !visible
                },
                State {
                    name: "withoutHandle"
                    when: visible && !_private.handleEnabled
                    PropertyChanges {
                        target: stackColumn
                        y: header.height
                    }
                    PropertyChanges {
                        target: scrollDecorator
                        y: scrollDecorator.headerHeight + flickable.contentY / stackColumn.height * scrollDecorator.workHeight
                    }
                },
                State {
                    name: "moving"
                    when: visible && _private.handleEnabled && flickable.moving
                    PropertyChanges {
                        target: stackColumn
                        y: header.height + flickable.height - _private.calcMinHeight
                    }
                    PropertyChanges {
                        target: scrollDecorator
                        y: {
                            if (pageHeight < _private.calcMaxHeight) {
                                return scrollDecorator.headerHeight
                            }
                            return scrollDecorator.headerHeight + Math.max(
                                            (flickable.contentY - stackColumn.y - flickable.y + header.height)
                                            / stackColumn.height * scrollDecorator.workHeight, 0
                                        )
                        }
                    }
                },
                State {
                    name: "minHeight"
                    when: visible && _private.handleEnabled && page.pageHeight == _private.calcMinHeight
                          && !flickable.moving && _private.calcHeightDiff !== 0
                    PropertyChanges {
                        target: stackColumn
                        y: Math.max(flickable.height - page.pageHeight, 0) + header.height
                    }
                    PropertyChanges {
                        target: scrollDecorator
                        y: scrollDecorator.headerHeight
                    }
                },
                State {
                    name: "midHeight"
                    when: visible && _private.handleEnabled && page.pageHeight < _private.calcMaxHeight
                          && page.pageHeight == flickable.height && !flickable.moving
                    PropertyChanges {
                        target: stackColumn
                        y: Math.max(flickable.contentY, 0) + header.height
                    }
                    PropertyChanges {
                        target: scrollDecorator
                        y: scrollDecorator.headerHeight
                    }
                },
                State {
                    name: "maxHeight"
                    when: visible && _private.handleEnabled && page.pageHeight === _private.calcMaxHeight
                          && !flickable.moving
                    PropertyChanges {
                        target: stackColumn
                        y: flickable.height - page.pageHeight + _private.calcHeightDiff + header.height
                    }
                    PropertyChanges {
                        target: scrollDecorator
                        y: scrollDecorator.headerHeight + (flickable.contentY - stackColumn.y + header.height)
                           / stackColumn.height * scrollDecorator.workHeight
                    }
                }
            ]
            spacing: Theme.paddingSmall
        }

        Rectangle {
            anchors {
                fill: header
                leftMargin: -border.width * 2
                rightMargin: -border.width * 2
                topMargin: -border.width * 2
            }
            scale: header.scale
            color: _parentPopup ? _parentPopup._backgroundColor : "transparent"
            border.width: Theme._lineWidth
            border.color: Qt.tint(Theme.rgba(color, 0.9), Theme.rgba(palette.primaryColor, 0.1))
            visible: header.visible
        }

        PopupMenuItem {
            id: header

            onClicked: {
                if (page._parentPopup) {
                    page._parentPopup._popPage(page)
                }
            }

            y: {
                if (!_private.handleEnabled) {
                    return Math.max(flickable.contentY, 0)
                }

                if (page.pageHeight == _private.calcMaxHeight && _private.calcMinHeight != _private.calcMaxHeight) {
                    return flickable.contentY - page.pageHeight + flickable.height - Theme.paddingSmall
                }

                return stackColumn.y - height - Theme.paddingSmall
            }
            z: 2
            width: parent.width
            height: text != "" ? Theme.dp(80) : (handle.height + handle.y)
            visible: text != ""
            _isHeaderItem: true
            icon.source: _isSubMenu ? "image://theme/icon-m-left" : ""
            _isClickable: _isSubMenu
        }
    }

    Rectangle {
        id: handle

        x: parent.width * 0.5 - handle.width * 0.5
        y: Theme.dp(8)
        z: 2
        width: Theme.dp(60)
        height: Theme.dp(4)
        radius: Theme.dp(2)
        color: Theme.rgba(palette.primaryColor, page.pageHeight >= _private.calcMaxHeight || !flickable.dragging ? 0.4 : 0.8)
        visible: _private.handleEnabled
        opacity: _inTransition ? 0 : 1

        Behavior on opacity {
            NumberAnimation {
                easing.type: Easing.InOutQuart
                duration: 200
            }
        }
    }

    Rectangle {
        id: scrollDecorator

        readonly property real headerHeight: header.visible ? header.height - (handle.visible ? Theme.paddingSmall : 0): 0
        readonly property real workHeight: pageHeight - _private.toolbarHeight - scrollDecorator.headerHeight

        anchors.right: page.right

        y: scrollDecorator.headerHeight
        width: Math.round(Theme.paddingSmall * 0.5)
        height: (scrollDecorator.workHeight - Theme.paddingSmall) * scrollDecorator.workHeight / stackColumn.height
        color: palette.primaryColor
        visible: _private.fullContentHeight > _private.calcMinHeight && opacity > 0
        opacity: scrollDecorator.workHeight > scrollDecorator.height ? 1 : 0

        Behavior on opacity { FadeAnimation { duration: 400 } }
    }

    Component.onCompleted: _defaultState()
}

