/****************************************************************************
**
** Copyright (c) 2021 - 2023 Open Mobile Platform LLC.
** Copyright (C) 2017 - 2021 Jolla Ltd.
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Sailfish.Pickers 1.0
import Nemo.FileManager 1.0

PickerDialog {
    id: page

    property alias path: fileModel.path
    property alias includeDirectories: fileModel.includeDirectories
    property alias includeHiddenFiles: fileModel.includeHiddenFiles
    property alias includeSystemFiles: fileModel.includeSystemFiles
    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias directorySort: fileModel.directorySort
    property alias nameFilters: fileModel.nameFilters
    property alias caseSensitivity: fileModel.caseSensitivity
    property alias showParentDirectory: gridView.showParentDirectory

    function updateSelectedFiles(selectionFlag) {
        for (var i = 0; i < fileModel.count; i++) {
            if (!fileModel.data(fileModel.index(i, 0), FileModel.IsDirRole)) {
                var path = fileModel.data(fileModel.index(i, 0), FileModel.AbsolutePathRole)
                var fileItem = {
                    'url': fileModel.data(fileModel.index(i, 0), FileModel.UrlRole),
                    'title': fileModel.data(fileModel.index(i, 0), FileModel.FileNameRole),
                    'lastModified': fileModel.data(fileModel.index(i, 0), FileModel.LastModifiedRole),
                    'filePath': fileModel.data(fileModel.index(i, 0), FileModel.AbsolutePathRole),
                    'fileName': fileModel.data(fileModel.index(i, 0), FileModel.FileNameRole),
                    'fileSize': fileModel.data(fileModel.index(i, 0), FileModel.SizeRole),
                    'mimeType': fileModel.data(fileModel.index(i, 0), FileModel.MimeTypeRole),
                    'selected': fileModel.data(fileModel.index(i, 0), FileModel.IsSelectedRole),
                    'contentType': ContentType.File,
                    "fileModel": fileModel
                }

                if (_selectedModel) {
                    if (selectionFlag) {
                        if (!_selectedModel.selected(path)) {
                            _selectedModel.append(fileItem)
                        }
                    } else {
                        if (_selectedModel.selected(path)) {
                            _selectedModel.remove(_selectedModel._indexInModel(fileItem))
                        }
                    }
                }
            }
        }
    }

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: _background ? gridView : __silica_applicationwindow_instance.contentItem
        targetPage: page
    }

    FileGridView {
        id: gridView

        property bool showParentDirectory
        readonly property int fileCount: fileModel.count - fileModel.countFolders

        function selectFileByModel() {
            for (var i = 0; i < fileModel.count; ++i) {
                var absolutePath = fileModel.fileNameAt(i)
                var selectedInSelectedModel = _selectedModel.selected(absolutePath)
                if (selectedInSelectedModel) {
                    fileModel.toggleSelectedFile(i)
                }
            }
        }

        anchors.fill: parent
        cellWidthForSmallScreen: parent.width
        currentIndex: -1
        selectionEnabled: true

        header: PickerDialogHeader {
            showBack: true
            selectedCount: page._selectedCount
            _glassOnly: _background
            fileSystemAccepted: true
        }

        model: FileModel {
            id: fileModel

            onPopulatedChanged: if (populated) gridView.selectFileByModel()

            path: StandardPaths.home
            active: page.status === PageStatus.Active
            includeDirectories: true
            sortBy: FileModel.SortByName
            directorySort: FileModel.SortDirectoriesBeforeFiles
            includeParentDirectory: parent.showParentDirectory
        }

        delegate: GridItem {
            id: fileDelegate

            function updateSelectedFile() {
                var fileItem = {
                    'url': model.url,
                    'title': model.fileName,
                    'lastModified': model.modified,
                    'filePath': model.absolutePath,
                    'fileName': model.fileName,
                    'fileSize': model.size,
                    'mimeType': model.mimeType,
                    'selected': model.isSelected,
                    'contentType': ContentType.File,
                    "fileModel": fileModel
                }

                if (_selectedModel) {
                    _selectedModel.update(fileItem)
                }
            }

            onClicked: {
                if (model.isDir) {
                    fileModel.clearSelectedFiles()
                    _selectedModel.clear()
                    var obj = pageStack.animatorPush('DirectoryDialog.qml', {
                                                         path: model.absolutePath,
                                                         includeHiddenFiles: fileModel.includeHiddenFiles,
                                                         sortBy: fileModel.sortBy,
                                                         sortOrder: fileModel.sortOrder,
                                                         directorySort: fileModel.directorySort,
                                                         nameFilters: fileModel.nameFilters,
                                                         caseSensitivity: fileModel.caseSensitivity,
                                                         _selectedModel: page._selectedModel,
                                                         acceptDestination: page.acceptDestination,
                                                         acceptDestinationAction: page.acceptDestinationAction,
                                                         _animationDuration: page._animationDuration,
                                                         _background: page._background,
                                                         _clearOnBackstep: false,
                                                         allowedOrientations: page.allowedOrientations,
                                                     })
                    obj.pageCompleted.connect(function(nextPage) {
                        nextPage.accepted.connect(function() {
                            page._dialogDone(DialogResult.Accepted)
                        })
                    })
                } else {
                    fileModel.toggleSelectedFile(index)
                    updateSelectedFile()
                }
            }

            contentWidth: fileItem.width
            contentHeight: fileItem.height
            _backgroundColor: "transparent"
            propagateComposedEvents: true
            menu: menuComponent

            FileGridItem {
                id: fileItem

                width: gridView.cellWidth
                chunkWidth: gridView.chunkWidth
                fileName: model.absolutePath
                size: model.size
                modified: model.modified
                displaySwitcherForDirectories : false
                selectionEnabled: gridView.fileCount > 0 && gridView.selectionEnabled
                selected: model.isSelected || _selectedModel.selected(model.absolutePath)
                onClicked: {
                    fileModel.toggleSelectedFile(index)
                    updateSelectedFile()
                }
            }

            Connections {
                target: gridView
                onWidthChanged: if (gridView.largeScreen && _menuItem && _menuItem.parent) _calculateMenuOffset()
            }

            Component {
                id: menuComponent

                ContextMenu {
                    MenuItem {
                        text: qsTrId("components_pickers-he-file_preview")
                        onClicked: Qt.openUrlExternally(model.absolutePath)
                    }
                }
            }
        }

        PickerPullDownMenu {
            sourceModel: fileModel
            currentPage: page
            currentView: gridView
        }

        ViewPlaceholder {
            text: {
                if (page.nameFilters.length) {
                    //: Empty state text if no files match the filter
                    //% "No files match filter"
                    return qsTrId("components_pickers-ph-no_matching_files")
                }
                //: Empty state text if the directory contains no content
                //% "Empty folder"
                return qsTrId("components_pickers-ph-empty_folder")
            }
            enabled: fileModel.populated && fileModel.count === 0 && !page.showParentDirectory
        }

        VerticalScrollDecorator {}
    }
}
