/*
 * Copyright (c) 2022 Open Mobile Platform LLC.
 * Copyright (c) 2018 Jolla Ltd.
 *
 * License: Proprietary
 */
import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Sailfish.Pickers 1.0
import Nemo.FileManager 1.0

BusyView {
    id: root

    property alias model: archiveModel
    property alias path: archiveModel.path
    property alias archiveFile: archiveModel.archiveFile
    property alias fileName: archiveModel.fileName
    property alias baseExtractionDirectory: archiveModel.archiveDirPath

    signal archiveExtracted(string containingFolder)
    signal showInfo(string info)

    function extractAllFiles(targetPath) {
        var target = _buildExtractionDirectory(false, true, model.baseName, targetPath)
        return archiveModel.extractAllFiles(target)
    }

    function extractFile(targetPath, fileName, isDir) {
        var targetDir = _buildExtractionDirectory(isDir, false, fileName, targetPath)
        return archiveModel.extractFile(fileName, targetDir)
    }

    function appendPath(fileName) {
        return archiveModel.appendPath(fileName)
    }

    function compressFiles(files, archiveName, targetPath, archiveType) {
        return archiveModel.compressFiles(files, archiveName, targetPath, archiveType)
    }

    function _buildExtractionDirectory(isDir, isArchive, dirName, targetPath) {
        var extractionDirectory = targetPath ? targetPath : baseExtractionDirectory

        if (isArchive || isDir) {
            return extractionDirectory + "/" + dirName
        } else {
            return extractionDirectory
        }
    }

    function selectedExtractPath(extract, fileName, isDir) {
        var obj = pageStack.animatorPush(folderPicker)
        obj.pageCompleted.connect(function(picker) {
            picker.selectedPathChanged.connect(function() {
                if (picker.selectedPath.length > 0) {
                    extract(picker.selectedPath, fileName, isDir)
                } else {
                    //% "Extraction path is incorrect"
                    showInfo(qsTrId("filemanager-la-extraction_path_incorrect"))
                }
            })
        })
    }

    Component {
        id: folderPicker

        FolderPickerPage {
            //% "Extract to"
            dialogTitle: qsTrId("filemanager-la-extract_to")
            showSystemFiles: false
            directorySort: SortingSettings.directorySort
            sortBy: SortingSettings.sortBy
            sortOrder: SortingSettings.sortOrder
        }
    }

    // Grace timer
    Timer {
        id: graceTimer
        interval: 500
        running: model.extracting || model.compressing
    }

    busy: model.extracting || model.compressing && !graceTimer.running
    enabled: busy
    //% "Extracting"
    text: model.extracting ? qsTrId("filemanager-la-extracting")
                             //% "Compressing"
                           : qsTrId("filemanager-la-compressing")

    ArchiveModel {
        id: archiveModel
        autoRename: true
        onErrorStateChanged: {
            switch (errorState) {
            case ArchiveModel.ErrorUnsupportedArchiveFormat:
                //% "Unsupported archive format"
                showInfo(qsTrId("filemanager-la-unsupported_archive_format"))
                break
            case ArchiveModel.ErrorArchiveNotFound:
                //% "Archive file is not found"
                showInfo(qsTrId("filemanager-la-archive_not_found"))
                break
            case ArchiveModel.ErrorArchiveOpenFailed:
                //% "Opening archive failed"
                showInfo(qsTrId("filemanager-la-opening_archive_failed"))
                break
            case ArchiveModel.ErrorArchiveExtractFailed:
                //% "Extract failed"
                showInfo(qsTrId("filemanager-la-extract_failed"))
                break
            case ArchiveModel.ErrorExtractingInProgress:
                //% "Extracting in progress"
                showInfo(qsTrId("filemanager-la-extracting_in_progress"))
                break
            case ArchiveModel.ErrorCompressingInProgress:
                //% "Compressing in progress"
                showInfo(qsTrId("filemanager-la-сompressing_in_progress"))
                break
            case ArchiveModel.ErrorFilesToCompressNotFound:
                //% "No files to compress"
                showInfo(qsTrId("filemanager-la-сompress_no_files"))
                break
            case ArchiveModel.ErrorInvalidArchiveName:
                //% "Invalid archive name"
                showInfo(qsTrId("filemanager-la-сompress_invalid_archive_name"))
                break
            case ArchiveModel.ErrorMissingDirForCreateArchive:
                //% "Missing directory for create archive"
                showInfo(qsTrId("filemanager-la-сompress_missing_directory"))
                break
            case ArchiveModel.ErrorMissingPathForCreateArchive:
                //% "Missing path for create archive"
                showInfo(qsTrId("filemanager-la-сompress_missing_path"))
                break
            case ArchiveModel.ErrorInvalidArchiveType:
                //% "Invalid archive type"
                showInfo(qsTrId("filemanager-la-сompress_invalid_archive_type"))
                break
            case ArchiveModel.ErrorFileCompresstFailed:
                //% "Compress file not found"
                showInfo(qsTrId("filemanager-la-сompress_file_not_found"))
                break
            }
        }

        onFilesExtracted: {
            if (isDir) {
                //% "Directory %1 extracted"
                showInfo(qsTrId("filemanager-la-directory_extracted").arg(entryName))
            } else if (entries && entries.length == 1) {
                //% "Extracted %1"
                showInfo(qsTrId("filemanager-la-file_extracted").arg(entryName))
            } else {
                //% "%1 extracted"
                showInfo(qsTrId("filemanager-la-archive_extracted").arg(fileName))
                root.archiveExtracted(path)
            }
        }

        onFilesCompressed: {
            //% "%1 created"
            showInfo(qsTrId("filemanager-la-file_compress_created").arg(archiveName))
        }
    }
}
