/*
 * Copyright (c) 2020 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */

#ifndef SAILFISH_MDM_DBUSJOURNAL_H
#define SAILFISH_MDM_DBUSJOURNAL_H

#include <QDBusMessage>
#include <QDBusPendingReply>
#include <QList>
#include <QMetaType>
#include <QObject>
#include <QPair>
#include <QString>
#include <QUuid>

class QDBusPendingCallWatcher;

namespace Sailfish {
namespace Mdm {

class DBusJournalPrivate;
class Q_DECL_EXPORT DBusJournal : public QObject
{
    Q_OBJECT
    Q_PROPERTY(int transactionsHandled READ transactionsHandled NOTIFY transactionsHandledChanged)
    Q_PROPERTY(int transactionsFailed READ transactionsFailed NOTIFY transactionsFailedChanged)
    Q_PROPERTY(int user READ user CONSTANT)

public:
    explicit DBusJournal(quint32 user, QObject *parent = nullptr);
    ~DBusJournal();

    static bool needsQueuedTransaction(quint32 user, const QString &service);
    static QString queuedTransaction(quint32 user, const QString &service, const QString &path,
                                     const QString &interface, const QString &method,
                                     const QVariantList &args,
                                     const QStringList &replyArgumentNames,
                                     quint32 timeout = -1,
                                     const QString &callbackService = QString(),
                                     const QString &callbackPath = QString(),
                                     const QString &callbackInterface = QString(),
                                     const QString &callbackMethod = QString());

    int user() const;
    int transactionsHandled() const;
    int transactionsFailed() const;

public slots:
    void fastForward();

signals:
    void transactionsHandledChanged();
    void transactionsFailedChanged();
    void transactionHandled(const QString &id);
    void transactionFailed(const QString &id, const QString &errorName, const QString &errorMessage);
    void journalRead(bool success);

private:
    static quint32 getCurrentServiceUser(const QString &service);
    static QMap<QString, QVariant> getDetails(QDBusPendingReply<> &reply, const QStringList &names);

    void readNextCall();
    void handleFinished(QDBusPendingCallWatcher *call, QUuid id, QDBusMessage cb, QStringList names);
    void handleCallback(QDBusPendingCallWatcher *call, QUuid id);
    bool callback(QDBusMessage &call, const QUuid &id, const QString &error, const QMap<QString, QVariant> &details);
    void streamEnded(bool success = true);

    QScopedPointer<DBusJournalPrivate> d_ptr;
};

}; // Mdm
}; // Sailfish

#endif // SAILFISH_MDM_DBUSJOURNAL_H
