/*
 * Copyright (c) 2017 - 2020 Jolla Ltd.
 * Copyright (c) 2022 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */

#ifndef SAILFISH_MDM_CERTIFICATE_H
#define SAILFISH_MDM_CERTIFICATE_H

#include <QObject>
#include <QSharedDataPointer>

QT_BEGIN_NAMESPACE
class QSslCertificate;
QT_END_NAMESPACE

namespace Sailfish {
namespace Mdm {

class Certificates;
class CertificateFilePrivate;
class Q_DECL_EXPORT CertificateFile
{
public:
    CertificateFile();
    CertificateFile(const CertificateFile &file);
    ~CertificateFile();

    CertificateFile &operator =(const CertificateFile &file);

    bool isNull() const;

    QString fileName() const;
    QList<QSslCertificate> certificates() const;

private:
    friend class Certificates;
    friend class CertificatesPrivate;

    inline CertificateFile(const QString &filePath, const QString &fileName);

    QExplicitlySharedDataPointer<CertificateFilePrivate> d;
};

class CertificatesPrivate;
class Q_DECL_EXPORT Certificates : public QObject
{
    Q_OBJECT
public:
    enum Result {
        Success,
        Error
    };

    enum ErrorType : int {
        NoError,
        InvalidCommand,
        InvalidFileName,
        EmptyFile,
        FileAlreadyExist,
        CantOpenFile,
        CantWriteFile,
        FailedVerify,
        FailedFinalizeFile,
        UpdateCaTrusIssue,
        DBusError
    };

    explicit Certificates(QObject *parent = nullptr);
    ~Certificates();

    QList<CertificateFile> installedFiles() const;
    QList<CertificateFile> blacklistedFiles() const;
    QList<QSslCertificate> certificates() const;

    void install(const QString &fileName, const QByteArray &data);
    void install(const QString &fileName, const QList<QSslCertificate> &certificates);
    void uninstall(const QString &fileName);

    void blacklist(const QString &fileName, const QByteArray &data);
    void blacklist(const QString &fileName, const QList<QSslCertificate> &certificates);
    void unblacklist(const QString &fileName);
    int lastErrorType();
    QString lastErrorMessage();

signals:
    void installFinished(const CertificateFile &file, Result result);
    void uninstallFinished(const CertificateFile &file, Result result);
    void blacklistFinished(const CertificateFile &file, Result result);
    void unblacklistFinished(const CertificateFile &file, Result result);

    void certificatesChanged();
    void installedFilesChanged();
    void blacklistedFilesChanged();

private:
    QScopedPointer<CertificatesPrivate> d;

    Q_DISABLE_COPY(Certificates)
};

} // namespace Mdm
} // namespace Sailfish

#endif // SAILFISH_MDM_CERTIFICATE_H
