/* This file is part of the Aurora OS SDK
*/

var merSdkTargets = "AuroraOS-5.2.0.45-aarch64 AuroraOS-5.2.0.45-armv7hl AuroraOS-5.2.0.45-x86_64";
var merSdkVmName = "Aurora Build Engine 5.2.0.45";
var merSdkVdiCapacityMB = "32000";
var merSdkVMSshPort = "2222";
var merSdkVMDBusPort = "7777";
var merSdkDir = "@TargetDir@/mersdk";
var sharedTargetsDir = "@TargetDir@/mersdk/targets";
var sharedSshDir = "@TargetDir@/mersdk/ssh";
var sharedConfigDir = "@TargetDir@/vmshare";
var sshKeyDir = sharedConfigDir + "/ssh/private_keys";

function dockerNetworkExists(networkName)
{
    const networkExistsCommand = ["network", "ls", "--filter", "name=" + networkName, "--quiet"];
    const result = installer.execute(installer.value("dockerPath"), networkExistsCommand)[0];
    return result.trim() !== "";
}

function dockerImageExists(imageName)
{
    const imageExistsCommand = ["image", "ls", imageName, "--quiet"];
    const result = installer.execute(installer.value("dockerPath"), imageExistsCommand)[0];
    return result.trim() !== "";
}

function componentExists(componentName)
{
    var retVal = false;
    installer.components().forEach(function (component) {
        if (component.name === componentName)
            retVal = true;
    });
    return retVal;
}

function dockerAvailable()
{
    return installer.value("dockerPath") && componentExists("org.merproject.mersdkdocker");
}

function dockerUsable()
{
    result = installer.execute(installer.value("dockerPath"), "ps");
    if (result.length < 2) {
        console.log("could not execute " + installer.value("dockerPath"));
        return false;
    }
    if (result[1] !== 0)
        console.log("docker returned " + result[1]);
    return result[1] === 0;
}

function setBuildEngineType(buildEngineType)
{
    installer.setValue("buildEngineType", buildEngineType);
    component.setValue("Dependencies", "");

    const merSdkDockerTag = installer.environmentVariable("USERNAME")
        || installer.environmentVariable("USER")
        || "latest";
    const merSdkDockerImage = "aurora-os-build-engine-5.2.0.45:" + merSdkDockerTag;
    installer.setValue("merSdkVmName", merSdkDockerImage);
    installer.setValue("merSdkVmUri", "sfdkvm:Docker#" + merSdkDockerImage);
    component.addDependency("org.merproject.mersdkdocker");

    installer.componentsToInstallNeedsRecalculation();
}

function Component()
{
    console.log("Component constructor: " + component.name);
    if (installer.value("abortInstaller"))
        return;

    component.setValue("DisplayName", qsTranslate("installscript", "Build Engine"));
    component.setValue("Description", component.value("DisplayName"));

    installer.setValue("merSdkVdiCapacityMB", merSdkVdiCapacityMB);

    installer.setValue("merSdkTargets", merSdkTargets);
    installer.setValue("merSdkVMSshPort", merSdkVMSshPort);
    installer.setValue("merSdkVMDBusPort", merSdkVMDBusPort);

    const setDirs = function() {
        installer.setValue("merSdkSharedTargetsDir", sharedTargetsDir);
        installer.setValue("merSdkSharedSshDir", sharedSshDir);
        installer.setValue("merSdkSharedConfigDir", sharedConfigDir);
        installer.setValue("merSdkSshKeyDir", sshKeyDir);
        installer.setValue("merSdkSshKeyFile", sshKeyDir + "/sdk");
        installer.setValue("merSdkSshKeyPublicFile", sshKeyDir + "/sdk.pub");
    }
    setDirs();
    installer.valueChanged.connect(this, function(key, value) {
            if (key === "TargetDir")
                setDirs();
    });

    if (installer.isInstaller()) {
        if (installer.value("rootLoaded")) {
            Component.prototype.ctor();
        } else {
            installer.valueChanged.connect(this, function(key, value) {
                if (key === "rootLoaded")
                    Component.prototype.ctor();
            });
        }
    } else {
        // updater and package manager case
        setBuildEngineType(installer.value("buildEngineType"));
    }
}

Component.prototype.ctor = function()
{
    const dockerInstallLink = qsTranslate("installscript", "<a href=\"https://developer.auroraos.ru/doc/sdk/app_development/setup/docker_install/\">Docker Install</a>");

    if (!dockerAvailable()) {
        installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                qsTranslate("installscript", "You need a working Docker installation. Please download from %1, install it and then rerun this installer.")
                .arg(dockerInstallLink));
        installer.setValue("abortInstaller", true);
        installer.setValue("existingInstallationWarning", true);
        return;
    }
    if (!dockerUsable()) {
        installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                qsTranslate("installscript", "<html><head/><body><p><span style='color:#ef2929;'>"
                    + "Docker is not running or misconfigured.<br/>"
                    + "Please follow the instructions in the "
                    + "<a href=\"https://developer.auroraos.ru/doc/sdk/app_development/setup\">Aurora SDK setup</a>"
                    + " and try again"
                    + "</span></p></body></html>"));
        installer.setValue("abortInstaller", true);
        installer.setValue("existingInstallationWarning", true);
        return;
    }

    setBuildEngineType("docker");
}

Component.prototype.createOperations = function()
{
    // Call the base createOperations
    component.createOperations();

    component.registerPathForUninstallation(sharedConfigDir, true);
    component.registerPathForUninstallation(sharedSshDir, true);

    component.addOperation("Mkdir", sharedConfigDir);
    component.addOperation("Mkdir", sharedSshDir);

    const sharedMerHomeDir = installer.value("sharedMerHomeDir");
    installer.performOperation("Mkdir", [sharedMerHomeDir]);

    addDockerOperations();
}

function addDockerOperations()
{
    const noop = installer.value("os") === "win"
        ? [ "cmd", "/c", "exit" ]
        : [ "true" ];
    const docker = installer.value("dockerPath");

    const userName = installer.environmentVariable("USERNAME")
        || installer.environmentVariable("USER");
    const networkName = "aurora-sdk-5.2.0.45:" + userName;

    Array.prototype.flat1 = function() {
        return this.reduce(function(acc, val) { return acc.concat(val) }, []);
    };

    const networkCreateCommand = [docker, "network", "create", "--driver", "bridge", networkName];
    const networkRemoveCommand = [docker, "network", "rm", networkName];

    const networkExists = dockerNetworkExists(networkName);

    if (networkExists) {
        const deleteNetworkTitle = qsTranslate("installscript", "A conflict of Docker networks has occurred.");
        const deleteNetworkMessage = qsTranslate("installscript", "The Docker network with name %1 already exists. Delete this network and create it again?").arg(networkName);
        var deleteNetwork = QMessageBox.question(
            "Delete Network?",
            deleteNetworkTitle,
            deleteNetworkMessage,
            QMessageBox.Yes | QMessageBox.No
        );

        if (deleteNetwork === QMessageBox.Yes) {
            component.addOperation("Execute", networkRemoveCommand);
            component.addOperation("Execute", [networkCreateCommand, "UNDOEXECUTE", networkRemoveCommand].flat1());
        } else {
            component.addOperation("Execute", [noop, "UNDOEXECUTE", networkRemoveCommand].flat1());
        }
    } else {
        component.addOperation("Execute", [networkCreateCommand, "UNDOEXECUTE", networkRemoveCommand].flat1());
    }

    component.addOperation("AppendFile", merSdkDir + "/Dockerfile",
`FROM scratch
ADD aurora.tar /
ADD --chown="${installer.value("uid")}" srv.tar / 
LABEL SharedInstall="${installer.value("TargetDir").replace(/\\/g, '/')}" \
      SharedHome="${installer.value("sharedMerHomeDir").replace(/\\/g, '/')}" \
      SharedSrc="${installer.value("workspaceDir").replace(/\\/g, '/')}" \
      SharedTarget="${installer.value("merSdkSharedTargetsDir").replace(/\\/g, '/')}" \
      SharedConfig="${installer.value("merSdkSharedConfigDir").replace(/\\/g, '/')}" \
      SharedSsh="${installer.value("merSdkSharedSshDir").replace(/\\/g, '/')}" \
      SshPort="${merSdkVMSshPort}" \
      DBusPort="${merSdkVMDBusPort}"
RUN chown -R --from=mersdk "${installer.value("uid")}" /etc/mersdk /home/mersdk; \
    usermod -u "${installer.value("uid")}" mersdk
CMD ["/usr/bin/setarch", "x86_64", "/sbin/init"]
`);
    component.addOperation("EnvironmentVariable", "DOCKER_BUILDKIT", "0", "false");

    const merSdkDockerTag = installer.environmentVariable("USERNAME")
        || installer.environmentVariable("USER")
        || "latest";
    const merSdkDockerImageName = "aurora-os-build-engine-5.2.0.45:" + merSdkDockerTag;
    const merSdkDockerContainerName = "/aurora-os-build-engine-5.2.0.45_" + merSdkDockerTag;
    const merSdkDockerVolumeName = "aurora-sdk-5.2.0.45-" + merSdkDockerTag;

    const imageBuildCommand = [docker, "build", "--tag", "@merSdkVmName@", "--quiet", merSdkDir];
    const imageRemoveCommand = [docker, "image", "rm", "--force", merSdkDockerImageName];
    const containerRemoveCommand = [docker, "container", "rm", "--force", merSdkDockerContainerName];
    const volumeRemoveCommand = [docker, "volume", "rm", "--force", merSdkDockerVolumeName];

    const imageExists = dockerImageExists(merSdkDockerImageName);

    if (imageExists) {
        const deleteImageTitle = qsTranslate("installscript", "A conflict of Docker images has occurred.");
        const deleteImageMessage = qsTranslate("installscript", "The Docker image with name %1 already exists. Delete this image and create it again?").arg(merSdkDockerImageName);
        var deleteImage = QMessageBox.question(
            "Delete Image?",
            deleteImageTitle,
            deleteImageMessage,
            QMessageBox.Yes | QMessageBox.No
        );

        if (deleteImage === QMessageBox.Yes) {
            component.addOperation("Execute", containerRemoveCommand);
            component.addOperation("Execute", volumeRemoveCommand);
            component.addOperation("Execute", imageRemoveCommand);
            component.addOperation("Execute", [imageBuildCommand, "UNDOEXECUTE", imageRemoveCommand].flat1());
        } else {
            component.addOperation("Execute", [noop, "UNDOEXECUTE", imageRemoveCommand].flat1());
        }
    } else {
        component.addOperation("Execute", [imageBuildCommand, "UNDOEXECUTE", imageRemoveCommand].flat1());
    }

    component.addOperation("Execute", [noop,
                                       "UNDOEXECUTE",
                                       volumeRemoveCommand].flat1());

    component.addOperation("Execute", [noop,
                                       "UNDOEXECUTE",
                                       containerRemoveCommand].flat1());

    component.addOperation("Delete", merSdkDir + "/aurora.tar");
    component.addOperation("Delete", merSdkDir + "/Dockerfile");

    // Run docker with volume optin for initialize volume. The operation can take a long time, so we perform it during installation
    const engineImageRunCommand = [docker, "run", "--rm", "-v", merSdkDockerVolumeName + ":/srv/mer", merSdkDockerImageName, "echo"];
    component.addOperation("Execute", engineImageRunCommand);
}

Component.prototype.createOperationsForArchive = function(archive)
{
    if (archive.endsWith("docker.7z"))
        component.addOperation("Extract", archive, "@TargetDir@");
}
