/****************************************************************************************
**
** SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.6
import Sailfish.Silica 1.0

SilicaControl {
    id: noticeItem

    property var notice

    property real horizontalMargin: Theme.horizontalPageMargin
    property real verticalMargin: Theme.paddingLarge

    x: {
        var left = Theme.horizontalPageMargin
        var right = parent.width - horizontalMargin - width

        var x
        switch (notice.anchor & (Notice.Left | Notice.Right)) {
        case Notice.Left:
            x = left
            break
        case Notice.Right:
            x = right
            break
        default:
            x = (parent.width - width) / 2
        }
        return Math.max(left, Math.min(right, x + notice.horizontalOffset))
    }

    y: {
        var top = Theme.horizontalPageMargin
        var bottom = parent.height - height - Theme.horizontalPageMargin

        var y
        switch (notice.anchor & (Notice.Top | Notice.Bottom)) {
        case Notice.Top:
            y = verticalMargin
            break
        case Notice.Bottom:
            y = parent.height - verticalMargin - height
            break
        default:
            y = (parent.height - height) / 2
        }
        return Math.max(top, Math.min(bottom, y + notice.verticalOffset))
    }

    implicitHeight: Math.max(Theme.itemSizeExtraSmall, label.implicitHeight)
    implicitWidth: Math.min(
                label.implicitWidth,
                parent.width - (2 * Theme.horizontalPageMargin))

    clip: label.implicitWidth > width

    highlighted: mouseArea.pressed

    MouseArea {
        id: mouseArea

        width: noticeItem.width
        height: noticeItem.height
        cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor

        onClicked: Notices._dismissCurrent()
    }

    Rectangle {
        width: noticeItem.width
        height: noticeItem.height
        color: Qt.tint(
                   Theme.rgba(noticeItem.palette.overlayBackgroundColor, Theme.opacityOverlay),
                   Theme.rgba(noticeItem.palette.highlightBackgroundColor, Theme.highlightBackgroundOpacity))
        radius: Theme.paddingSmall
    }

    Label {
        id: label

        width: Math.max(noticeItem.width, implicitWidth)
        height: noticeItem.height

        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        leftPadding: Theme.paddingLarge
        rightPadding: Theme.paddingLarge
        topPadding: Theme.paddingSmall
        bottomPadding: Theme.paddingSmall

        text: noticeItem.notice.text

        font.pixelSize: Theme.fontSizeExtraSmall

        SequentialAnimation on x {
            running: label.width > noticeItem.width
            loops: Animation.Infinite

            PauseAnimation {
                duration: 1500
            }
            SmoothedAnimation {
                velocity: 100 * Theme.pixelRatio
                duration: -1
                from: 0
                to: -label.width + noticeItem.width
                easing.type: Easing.Linear
            }
            PauseAnimation {
                duration: 2000
            }
            PropertyAction {
                value: 0
            }
        }
    }
}
