/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013-2019 Jolla Ltd.
** SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import "private"

TextBase {
    id: textField

    property alias text: preeditText.text
    property alias textWidth: textInput.width
    property alias readOnly: textInput.readOnly
    property int inputMethodHints
    property alias inputMethodComposing: textInput.inputMethodComposing
    property alias validator: proxyValidator.validator
    property alias echoMode: textInput.echoMode
    property alias cursorPosition: textInput.cursorPosition
    property alias wrapMode: textInput.wrapMode
    property alias selectedText: textInput.selectedText
    property alias selectionStart: textInput.selectionStart
    property alias selectionEnd: textInput.selectionEnd
    property bool acceptableInput: textInput.acceptableInput
    property alias passwordCharacter: textInput.passwordCharacter
    property alias passwordMaskDelay: textInput.passwordMaskDelay
    property alias maximumLength: textInput.maximumLength
    property alias length: textInput.length
    property alias strictValidation: proxyValidator.strictValidation

    property bool _cursorBlinkEnabled: true
    property real _minimumWidth: textField.width - Theme.paddingSmall - textField._totalLeftMargins
                                 - textField._totalRightMargins - textField._rightItemWidth
    property alias _editorFont: textInput.font
    property bool __silica_textfield: true

    onHorizontalAlignmentChanged: {
        if (explicitHorizontalAlignment) {
            textInput.horizontalAlignment = horizontalAlignment
        }
    }
    onExplicitHorizontalAlignmentChanged: {
        if (explicitHorizontalAlignment) {
            textInput.horizontalAlignment = horizontalAlignment
        } else {
            textInput.horizontalAlignment = undefined
        }
    }

    _editor: textInput
    _flickableDirection: Flickable.VerticalFlick
    errorHighlight: textInput.touched && !textField.acceptableInput

    onReadOnlyChanged: _updateBackground()

    TextInput {
        id: textInput
        objectName: "textEditor"

        // Workaround for cursor delegate unable to reference directly to "textField"
        // Should be fixed in Qt5. To be verified...
        property alias cursorColor: textField.cursorColor
        property bool touched

        onActiveFocusChanged: if (!activeFocus) touched = true
        onTextChanged: if (activeFocus) touched = true

        onHorizontalAlignmentChanged: textField.setImplicitHorizontalAlignment(horizontalAlignment)

        x: -parent.contentX + textField.textLeftPadding
        y: -parent.contentY + textField.textTopPadding
        width: textField._minimumWidth
        focus: true
        activeFocusOnPress: false
        passwordCharacter: "\u2022"
        color: textField.color
        selectionColor: Theme.rgba(textField.palette.highlightColor, 0.2)
        selectedTextColor: textField.palette.highlightColor
        font: textField.font
        cursorDelegate: Cursor {
            color: textField.cursorColor
            preedit: preeditText
            _blinkEnabled: textField._cursorBlinkEnabled
        }
        validator: proxyValidator.validator ? proxyValidator : null

        // JB#45985 and QTBUG-37850: Qt was changed to mess up with virtual keyboard state when enter key
        // is handled. Work around by always forcing multiline hint for this single line entry
        inputMethodHints: textField.inputMethodHints | Qt.ImhMultiLine

        ProxyValidator {
            id: proxyValidator
        }

        PreeditText {
            id: preeditText

            onTextChanged: {
                textField._fixupScrollPosition()
                if (activeFocus) {
                    textInput.touched = true
                }
            }
        }

        states: State {
            when: textInput.wrapMode === TextInput.NoWrap
            PropertyChanges {
                target: textField
                _flickableDirection: Flickable.HorizontalFlick
                _singleLine: true
            }
            PropertyChanges {
                target: textInput
                width: Math.max(_minimumWidth, implicitWidth)
            }
        }
    }
}
