/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013 Jolla Ltd
** SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0

SilicaItem {
    id: root

    property real maximumValue: 1.0
    property real minimumValue
    property real value
    property real progressValue
    property bool indeterminate
    property bool glowEffect: true
    property string valueText
    property alias label: labelText.text
    property real leftMargin: Math.round(Screen.width/8)
    property real rightMargin: Math.round(Screen.width/8)
    readonly property real barCenterY: backgroundLine.y
    readonly property bool _running: visible && opacity > 0.0 && Qt.application.active
                                     && (glowEffect || indeterminate)
    readonly property real normalizedValue: (Math.max(minimumValue, Math.min(maximumValue, value)) - minimumValue)
                                            / (maximumValue - minimumValue)

    onNormalizedValueChanged: {
        if (progressValue > normalizedValue) {
            smoothedAnimation.enabled = false
            progressValue = normalizedValue
            smoothedAnimation.enabled = true
        } else {
            progressValue = normalizedValue
        }
    }

    implicitHeight: {
        const neededHeight = valueTextPlaceholder.anchors.bottomMargin + backgroundLine.anchors.bottomMargin
                           + labelText.anchors.bottomMargin + valueTextPlaceholder.height
                           + backgroundLine.height + labelText.height

        if (valueText && label) {
            return Math.max(Theme.itemSizeLarge, neededHeight)
        }

        if ((valueText && !label) || (!valueText && label)) {
            return Math.max(Theme.itemSizeMedium, neededHeight)
        }

        return Theme.itemSizeSmall
    }

    Behavior on progressValue {
        id: smoothedAnimation

        SmoothedAnimation {
            velocity: 0.2
            duration: 200
        }
    }

    Item {
        id: contentColumn

        anchors {
            leftMargin: root.leftMargin
            left: parent.left
        }

        width: parent.width - leftMargin - rightMargin
        height: parent.height

        Text {
            id: valueTextPlaceholder

            anchors {
                horizontalCenter: parent.horizontalCenter
                bottom: backgroundLine.top
                bottomMargin: Theme.dp(15)
            }

            font.pixelSize: Theme.fontSizeMedium
            font.family: Theme.fontFamily
            color: palette.primaryColor
            textFormat: Text.PlainText
            text: root.valueText
            visible: text
        }

        Rectangle {
            id: backgroundLine

            anchors {
                horizontalCenter: parent.horizontalCenter
                verticalCenter: !label ? parent.verticalCenter : undefined
                bottom: label ? labelText.top : undefined
                bottomMargin: label ? Theme.dp(29) : 0
            }

            width: parent.width
            height: Theme.dp(6)
            radius: height * 0.5
            color: Theme.rgba(palette.primaryColor, Theme.opacityFaint)

            ShaderEffect {
                id: shader

                property color dashColor: indeterminate ? "transparent" : palette.highlightColor
                property color progressDashColor: {
                    if (!root._running) {
                        return "transparent"
                    }

                    if (indeterminate) {
                        return palette.highlightColor
                    }

                    return Qt.tint(palette.highlightColor, Theme.rgba(palette.highlightBackgroundColor, 0.5))
                }

                property real firstDashStartPos
                property real firstDashEndPos
                property real secondDashStartPos
                property real secondDashEndPos
                property real offset: width / 6
                property real radius: height * 0.5
                property real progressValue: indeterminate ? 1.0 : root.progressValue

                width: parent.width
                height: backgroundLine.height

                fragmentShader: "
                    precision highp float;

                    varying mediump vec2 qt_TexCoord0;

                    uniform highp float firstDashStartPos;
                    uniform highp float firstDashEndPos;
                    uniform highp float secondDashStartPos;
                    uniform highp float secondDashEndPos;
                    uniform highp float height;
                    uniform highp float width;
                    uniform highp float progressValue;
                    uniform highp float radius;
                    uniform lowp vec4 dashColor;
                    uniform lowp vec4 progressDashColor;

                    float dashDistance(vec2 pos, vec2 dashCenter, vec2 dashSize, float dashRadius) {
                        vec2 d = abs(dashCenter - pos) - (dashSize - vec2(dashRadius * 2.0)) * 0.5;
                        return length(max(d,0.0)) - dashRadius;
                    }

                    void main() {
                        vec2 pos = qt_TexCoord0.xy * vec2(width, height);

                        vec2 progressSize = vec2(width * progressValue, height);
                        vec2 progressCenter = vec2(progressSize.x * 0.5, progressSize.y * 0.5);
                        float progressDistance = dashDistance(pos, progressCenter, progressSize, radius);

                        vec2 firstDashSize = vec2(firstDashEndPos - firstDashStartPos, height);
                        vec2 firstDashCenter = vec2(firstDashStartPos + firstDashSize.x * 0.5, firstDashSize.y * 0.5);
                        float firstDashDistance = dashDistance(pos, firstDashCenter, firstDashSize, radius);

                        vec2 secondDashSize = vec2(secondDashEndPos - secondDashStartPos, height);
                        vec2 secondDashCenter = vec2(secondDashStartPos + secondDashSize.x * 0.5, secondDashSize.y * 0.5);
                        float secondDashDistance = dashDistance(pos, secondDashCenter, secondDashSize, radius);

                        vec4 progressColorRes = dashColor * (1.0 - clamp(progressDistance + 0.5, 0.0, 1.0));
                        vec4 progressDashColorRes = progressDashColor * (1.0 - clamp(firstDashDistance + 0.5, 0.0, 1.0)) +
                                                    progressDashColor * (1.0 - clamp(secondDashDistance + 0.5, 0.0, 1.0));

                        progressColorRes = mix(dashColor, progressDashColorRes, progressDashColorRes.a) * (1.0 - clamp(progressDistance + 0.5, 0.0, 1.0));

                        gl_FragColor = progressColorRes;
                    }
                "

                PropertyAnimation {
                    easing.type: Easing.Bezier
                    easing.bezierCurve: [0.6, 0.3, 0.25, 0.5, 1, 1]
                    target: shader
                    property: "firstDashStartPos"
                    running: root._running
                    loops: Animation.Infinite
                    from: 0.0 - shader.offset
                    to: shader.width
                    duration: 2000
                }

                PropertyAnimation {
                    easing.type: Easing.Bezier
                    easing.bezierCurve: [0.45, 0.4, 0.25, 0.6, 1, 1]
                    target: shader
                    property: "firstDashEndPos"
                    running: root._running
                    loops: Animation.Infinite
                    from: 0.0
                    to: shader.width + shader.offset
                    duration: 2000
                }

                SequentialAnimation {
                    running: root._running

                    PropertyAction {
                        target: shader
                        property: "secondDashStartPos"
                        value: 0.0
                    }

                    PropertyAction {
                        target: shader
                        property: "secondDashEndPos"
                        value: 0.0
                    }

                    PauseAnimation {
                        duration: 1000
                    }

                    ParallelAnimation {
                        PropertyAnimation {
                            easing.type: Easing.Bezier
                            easing.bezierCurve: [0.6, 0.3, 0.25, 0.5, 1, 1]
                            target: shader
                            property: "secondDashStartPos"
                            loops: Animation.Infinite
                            from: 0.0 - shader.offset
                            to: shader.width
                            duration: 2000
                        }

                        PropertyAnimation {
                            easing.type: Easing.Bezier
                            easing.bezierCurve: [0.45, 0.4, 0.25, 0.6, 1, 1]
                            target: shader
                            property: "secondDashEndPos"
                            loops: Animation.Infinite
                            from: 0.0
                            to: shader.width + shader.offset
                            duration: 2000
                        }
                    }
                }
            }
        }

        Label {
            id: labelText

            anchors {
                bottom: parent.bottom
                bottomMargin: text ? Theme.dp(2) : 0
            }

            width: Math.min(contentWidth, parent.width)
            truncationMode: TruncationMode.Fade
            visible: text.length
            font.pixelSize: Theme.fontSizeExtraSmall
            font.family: Theme.fontFamily
            color: palette.secondaryColor
            textFormat: Text.PlainText
        }
    }
}
