/**
 * SPDX-FileCopyrightText: 2012 Digia Plc and/or its subsidiary(-ies)
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPLv2
 */

#ifndef QGALLERYABSTRACTREQUEST_H
#define QGALLERYABSTRACTREQUEST_H

#include "qgalleryglobal.h"

#include <QtCore/qobject.h>
#include <QtCore/qstringlist.h>

QT_BEGIN_HEADER
QT_BEGIN_NAMESPACE_DOCGALLERY

class QAbstractGallery;
class QGalleryAbstractResponse;
class QGalleryAbstractRequestPrivate;

class Q_GALLERY_EXPORT QGalleryAbstractRequest : public QObject
{
    Q_OBJECT

    Q_ENUMS(State)
    Q_ENUMS(RequestType)

    Q_PROPERTY(QAbstractGallery* gallery READ gallery WRITE setGallery NOTIFY galleryChanged)
    Q_PROPERTY(bool supported READ isSupported NOTIFY supportedChanged)
    Q_PROPERTY(State state READ state NOTIFY stateChanged)
    Q_PROPERTY(int error READ error NOTIFY errorChanged)
    Q_PROPERTY(QString errorString READ errorString NOTIFY errorChanged)
    Q_PROPERTY(int currentProgress READ currentProgress NOTIFY progressChanged)
    Q_PROPERTY(int maximumProgress READ maximumProgress NOTIFY progressChanged)

public:
    enum State {
        Inactive,
        Active,
        Canceling,
        Canceled,
        Idle,
        Finished,
        Error
    };

    enum RequestError {
        NoError,
        NoGallery,
        NotSupported,
        GalleryError = 100
    };

    enum RequestType {
        QueryRequest,
        ItemRequest,
        TypeRequest
    };

    explicit QGalleryAbstractRequest(RequestType type, QObject *parent = Q_NULLPTR);
    explicit QGalleryAbstractRequest( QAbstractGallery *gallery, RequestType type, QObject *parent = Q_NULLPTR);
    ~QGalleryAbstractRequest();

    QAbstractGallery *gallery() const;
    void setGallery(QAbstractGallery *gallery);

    bool isSupported() const;

    RequestType type() const;
    State state() const;

    int error() const;
    QString errorString() const;

    int currentProgress() const;
    int maximumProgress() const;

    bool waitForFinished(int msecs);

public Q_SLOTS:
    void execute();
    void cancel();
    void clear();

Q_SIGNALS:
    void galleryChanged();
    void supportedChanged();
    void finished();
    void canceled();
    void error(int error, const QString &errorString);
    void stateChanged(QGalleryAbstractRequest::State state);
    void errorChanged();
    void progressChanged(int current, int maximum);
    void dataAvailableChanged(bool value);

protected:
    virtual void setResponse(QGalleryAbstractResponse *response) = 0;

    explicit QGalleryAbstractRequest(QGalleryAbstractRequestPrivate &dd, QObject *parent);

    QScopedPointer<QGalleryAbstractRequestPrivate> d_ptr;

private:
    Q_DECLARE_PRIVATE(QGalleryAbstractRequest)
    Q_PRIVATE_SLOT(d_func(), void _q_finished())
    Q_PRIVATE_SLOT(d_func(), void _q_canceled())
    Q_PRIVATE_SLOT(d_func(), void _q_resumed())
    Q_PRIVATE_SLOT(d_func(), void _q_progressChanged(int, int))
};

QT_END_NAMESPACE_DOCGALLERY
QT_END_HEADER

#endif
