/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QMap>
#include <QObject>
#include <QScopedPointer>
#include <QVariant>

namespace Aurora {
namespace Mdm {
namespace Connectivity {

class NetworkServicePrivate;
class Q_DECL_EXPORT NetworkService : public QObject
{
    Q_OBJECT

    Q_PROPERTY(bool valid READ isValid NOTIFY validChanged)
    Q_PROPERTY(QString name READ name NOTIFY nameChanged)
    Q_PROPERTY(QString state READ state NOTIFY stateChanged)
    Q_PROPERTY(QString type READ type NOTIFY typeChanged)
    Q_PROPERTY(QString error READ error NOTIFY errorChanged)
    Q_PROPERTY(QStringList security READ security NOTIFY securityChanged)
    Q_PROPERTY(SecurityType securityType READ securityType NOTIFY securityTypeChanged)
    Q_PROPERTY(uint strength READ strength NOTIFY strengthChanged)
    Q_PROPERTY(bool favorite READ favorite NOTIFY favoriteChanged)
    Q_PROPERTY(bool autoConnect READ autoConnect WRITE setAutoConnect NOTIFY autoConnectChanged)
    Q_PROPERTY(QString path READ path WRITE setPath NOTIFY pathChanged)
    Q_PROPERTY(QVariantMap ipv4 READ ipv4 NOTIFY ipv4Changed)
    Q_PROPERTY(QVariantMap ipv4Config READ ipv4Config WRITE setIpv4Config NOTIFY ipv4ConfigChanged)
    Q_PROPERTY(QVariantMap ipv6 READ ipv6 NOTIFY ipv6Changed)
    Q_PROPERTY(QVariantMap ipv6Config READ ipv6Config WRITE setIpv6Config NOTIFY ipv6ConfigChanged)
    Q_PROPERTY(QStringList nameservers READ nameservers NOTIFY nameserversChanged)
    Q_PROPERTY(QStringList nameserversConfig READ nameserversConfig WRITE setNameserversConfig NOTIFY nameserversConfigChanged)
    Q_PROPERTY(QStringList domains READ domains NOTIFY domainsChanged)
    Q_PROPERTY(QStringList domainsConfig READ domainsConfig WRITE setDomainsConfig NOTIFY domainsConfigChanged)
    Q_PROPERTY(QVariantMap proxy READ proxy NOTIFY proxyChanged)
    Q_PROPERTY(QVariantMap proxyConfig READ proxyConfig WRITE setProxyConfig NOTIFY proxyConfigChanged)
    Q_PROPERTY(QVariantMap ethernet READ ethernet NOTIFY ethernetChanged)
    Q_PROPERTY(bool roaming READ roaming NOTIFY roamingChanged)
    Q_PROPERTY(bool connected READ connected NOTIFY connectedChanged)
    Q_PROPERTY(QStringList timeservers READ timeservers NOTIFY timeserversChanged)
    Q_PROPERTY(QStringList timeserversConfig READ timeserversConfig WRITE setTimeserversConfig NOTIFY timeserversConfigChanged)
    Q_PROPERTY(bool saved READ saved NOTIFY savedChanged)
    Q_PROPERTY(bool managed READ managed NOTIFY managedChanged)
    Q_PROPERTY(bool connecting READ connecting NOTIFY connectingChanged)
    Q_PROPERTY(bool available READ available NOTIFY availableChanged)
    Q_PROPERTY(OwnerType ownerType READ ownerType NOTIFY ownerTypeChanged)
    Q_PROPERTY(QString lastConnectError READ lastConnectError NOTIFY lastConnectErrorChanged)

public:
    enum OwnerType {
        OwnerUser = 1,
        OwnerMdm = 2,
        OwnerAll = 3,
    };
    Q_ENUM(OwnerType);

    enum SecurityType {
        SecurityUnknown,
        SecurityNone,
        SecurityWEP,
        SecurityPSK,
        SecurityIEEE802
    };
    Q_ENUM(SecurityType);

    friend class NetworkManagerPrivate;

    NetworkService(const QString &path, const QVariantMap &properties, QObject* parent = nullptr);
    virtual ~NetworkService();

    QString name() const;
    QString type() const;
    QString state() const;
    QString error() const;
    QStringList security() const;
    SecurityType securityType() const;
    bool autoConnect() const;
    uint strength() const;
    bool favorite() const;
    QString path() const;
    QVariantMap ipv4() const;
    QVariantMap ipv4Config() const;
    QVariantMap ipv6() const;
    QVariantMap ipv6Config() const;
    QStringList nameservers() const;
    QStringList nameserversConfig() const;
    QStringList domains() const;
    QStringList domainsConfig() const;
    QVariantMap proxy() const;
    QVariantMap proxyConfig() const;
    QVariantMap ethernet() const;
    bool roaming() const;
    OwnerType ownerType() const;

    bool isValid() const;
    bool connected() const;
    bool available() const;
    bool managed() const;
    bool saved() const;
    bool connecting() const;
    QString lastConnectError() const;

    QStringList timeservers() const;
    QStringList timeserversConfig() const;
    void setTimeserversConfig(const QStringList &servers);
    void setPath(const QString &path);

    void requestConnect();
    void requestDisconnect();
    void remove();
    void setAutoConnect(bool autoconnect);
    void setIpv4Config(const QVariantMap &ipv4);
    void setIpv6Config(const QVariantMap &ipv6);
    void setNameserversConfig(const QStringList &nameservers);
    void setDomainsConfig(const QStringList &domains);
    void setProxyConfig(const QVariantMap &proxy);
    void resetCounters();

Q_SIGNALS:
    void validChanged();
    void nameChanged();
    void stateChanged();
    void errorChanged();
    void securityChanged();
    void securityTypeChanged();
    void strengthChanged();
    void favoriteChanged();
    void autoConnectChanged();
    void pathChanged();
    void ipv4Changed();
    void ipv4ConfigChanged();
    void ipv6Changed();
    void ipv6ConfigChanged();
    void nameserversChanged();
    void nameserversConfigChanged();
    void domainsChanged();
    void domainsConfigChanged();
    void proxyChanged();
    void proxyConfigChanged();
    void ethernetChanged();
    void connectRequestFailed();
    void typeChanged();
    void roamingChanged();
    void timeserversChanged();
    void timeserversConfigChanged();
    void serviceConnectionStarted();
    void serviceDisconnectionStarted();
    void connectedChanged();
    void propertiesReady();
    void managedChanged();
    void connectingChanged();
    void savedChanged();
    void availableChanged();
    void ownerTypeChanged();
    void lastConnectErrorChanged();
protected:
    NetworkService(NetworkServicePrivate* p, QObject* parent = nullptr);
    QScopedPointer<NetworkServicePrivate> d_ptr;

private:
    Q_DISABLE_COPY(NetworkService)
};

} // namespace Connectivity
} // namespace Mdm
} // namespace Aurora
