﻿/****************************************************************************************
**
** SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import "private/Util.js" as UtilAurora
import Aurora.Controls 1.0
import Aurora.Controls.private 1.0
import QtGraphicalEffects 1.0

SilicaItem {
    id: appBar

    default property alias content: appBarContainer.children

    property bool _bottomPosition
    property bool monochrome
    property bool background
    property bool headerClickable
    property bool divider: true
    readonly property bool opened: _openStatus == 1 && visible
    property string headerText
    property string subHeaderText
    property real maximumHeaderWidth
    property real minimumHeaderWidth
    property alias iconContainer: iconContainerObject
    readonly property real _cursorMinWidth: Theme.paddingSmall
    readonly property real _cursorStartX: {
        if (_bottomPosition) {
            return _leftCutoutInset + (_firstVisibleSearchField ? Theme.paddingLarge : Theme.horizontalPageMargin)
        }
        if (headerRow.visible && headerRow.width > 0) {
            return headerRow.anchors.leftMargin + Theme.paddingMedium + headerRow.width
        }
        return _leftCutoutInset + (_firstVisibleSearchField
                                   ? Theme.paddingLarge
                                   : (_landscape && !Screen._isTablet ? Theme.horizontalPageMargin
                                                                      : Theme.dp(4)))
    }

    readonly property real _contentContainerWidth: appBarContainer.width
                                                   + (headerClickable && headerRow.visible
                                                      ? headerRow.width + Theme.paddingMedium : 0)

    property var _selectedItem

    property QtObject _ngfEffectSelect
    property QtObject _ngfEffectHighlight

    property real _openStatus: 1

    readonly property SplitView _splitView: UtilAurora.findParentWithProperty(appBar, "__auroracontrols_splitview")
    readonly property bool _splitViewhasMoreOneSplit: _splitView && _splitView.lastActiveItem != _splitView.firstActiveItem

    readonly property var _tabItem: UtilAurora.findParentWithProperty(appBar, "__silica_tabitem")
    readonly property bool _tabItemActive: _tabItem ? _tabItem.isCurrentItem : true

    readonly property Page _page: UtilAurora.findParentWithProperty(appBar, "__silica_page")
    readonly property bool _pageActive: _page ? _page.status === PageStatus.Active : true

    readonly property bool _active: _pageActive && appBar.visible && appBarBackground.opacity > 0

    property real _availableSpace

    readonly property string __spacerSpecialProperty: "__auroracontrols_appbarSpacer"
    readonly property string __searchSpecialProperty: "__auroracontrols_appbarSearch"

    property bool _firstVisibleSearchField
    property bool _lastVisibleSearchField

    readonly property bool _landscape: SafeZoneRect._orientation & Orientation.LandscapeMask

    readonly property real _leftCutoutInset: {
        if (__silica_applicationwindow_instance.displayMode !== ApplicationDisplayMode.FillScreen
                || !_landscape
                || (_landscape && _splitViewhasMoreOneSplit)) {
            return 0
        }

        const cutOutRect = Private.CutoutArea.cutoutRect

        if (cutOutRect.y < Screen.height / 2 && (SafeZoneRect._orientation & Orientation.Landscape)) {
            if (_bottomPosition && cutOutRect.x > Theme._itemSizeHeader
                    || !_bottomPosition && cutOutRect.x + cutOutRect.width < Screen.width - Theme._itemSizeHeader) {
                return 0
            }

            return cutOutRect.y + cutOutRect.height

        } else if (cutOutRect.y >= Screen.height / 2 && (SafeZoneRect._orientation & Orientation.LandscapeInverted)) {
            if (!_bottomPosition && cutOutRect.x > Theme._itemSizeHeader
                    || _bottomPosition && cutOutRect.x + cutOutRect.width < Screen.width - Theme._itemSizeHeader) {
                return 0
            }

            return Screen.height - cutOutRect.y + cutOutRect.height
        }
        return 0
    }

    readonly property real _rightCutoutInset: {
        if (__silica_applicationwindow_instance.displayMode !== ApplicationDisplayMode.FillScreen
                || !_landscape
                || (_landscape && _splitViewhasMoreOneSplit)) {
            return 0
        }

        const cutOutRect = Private.CutoutArea.cutoutRect

        if (cutOutRect.y < Screen.height / 2 && (SafeZoneRect._orientation & Orientation.LandscapeInverted)) {
            if (!_bottomPosition && cutOutRect.x > Theme._itemSizeHeader
                    || _bottomPosition && cutOutRect.x + cutOutRect.width < Screen.width - Theme._itemSizeHeader) {
                return 0
            }

            return cutOutRect.y + cutOutRect.height
        } else if (cutOutRect.y >= Screen.height / 2 && (SafeZoneRect._orientation & Orientation.Landscape)) {
            if (_bottomPosition && cutOutRect.x > Theme._itemSizeHeader
                    || !_bottomPosition && cutOutRect.x + cutOutRect.width < Screen.width - Theme._itemSizeHeader) {

                return 0
            }

            return Screen.height - cutOutRect.y + cutOutRect.height
        }
        return 0
    }

    signal headerClicked()

    function _updateActiveButtons() {
        var currentCount = headerClickable && headerRow.visible ? 1 : 0

        for (var i in content) {
            const child = content[i]

            if (child.hasOwnProperty(__spacerSpecialProperty)
                    || !child.visible
                    || child.width === 0
                    || !child.enabled) {
                continue
            }
            currentCount++
        }
        pulleyGesture.activeButtonsCount = currentCount
    }

    function _updateComponentsWidth() {
        var visibleObjectsWidth = 0
        var spacersCount = 0
        var minimalFilledWidth = 0
        var minimalFilledArray = []
        // These properties represent left/right margins of appBarContainer.
        var hasSpacerBeforeSearchField = false
        var hasSpacerAfterSearchField = false

        var visibleObjBeforeSearchField = 0
        var visibleObjAfterSearchField = 0

        var hasSearchField = false

        for (var i in content) {
            const obj = content[i]

            if (obj.hasOwnProperty(__spacerSpecialProperty)
                    && obj.visible
                    && obj.fixedWidth === 0) {
                spacersCount += 1

                if (hasSearchField) {
                    hasSpacerAfterSearchField = true
                } else {
                    hasSpacerBeforeSearchField = true
                }

                continue
            }

            if (!obj.visible || obj.width === 0) {
                continue
            }

            if (obj.hasOwnProperty(__searchSpecialProperty) && !hasSearchField) {
               hasSearchField = true
            }

            if (obj.hasOwnProperty("__fillWidth") && obj.__fillWidth) {
                minimalFilledWidth += obj.minimalWidth
                minimalFilledArray.push(obj.minimalWidth)

                continue
            }

            if (!obj.hasOwnProperty(__searchSpecialProperty)) {
                if (hasSearchField) {
                    visibleObjAfterSearchField += 1
                } else {
                    visibleObjBeforeSearchField += 1
                }
            }

            visibleObjectsWidth += obj.width
        }

        if (hasSearchField && minimalFilledArray.length > 0) {
            _firstVisibleSearchField = visibleObjBeforeSearchField === 0
            _lastVisibleSearchField = visibleObjAfterSearchField === 0
        } else if (hasSearchField) {
            _firstVisibleSearchField = !hasSpacerBeforeSearchField
            _lastVisibleSearchField = !hasSpacerAfterSearchField
        } else {
            _firstVisibleSearchField = false
            _lastVisibleSearchField = false
        }

        var availaibleSpace = appBarContainer.width - visibleObjectsWidth

        if (headerRow.visible && headerRow.width > 0) {
            availaibleSpace += headerRow.width
        }
        const leftPadding = _landscape && !Screen._isTablet ? Theme.horizontalPageMargin : Theme.dp(4)
        if (minimalFilledWidth) {
            appBarContainer._spacersWidth = 0
            if (headerRow.visible && availaibleSpace - minimalFilledWidth < headerRow.targetWidth) {
                if (headerRow.width > 0) {
                    _availableSpace = availaibleSpace - minimalFilledWidth
                } else {
                    _availableSpace = availaibleSpace
                            - minimalFilledWidth
                            - Theme.horizontalPageMargin
                            - Theme.paddingMedium
                            + leftPadding
                }

                appBarContainer._fillComponentsWidth = 0
            } else {
                if (headerRow.width <= 0 && headerRow.targetWidth > 0) {
                    availaibleSpace -= Theme.horizontalPageMargin
                            + Theme.paddingMedium
                            - leftPadding
                            + headerRow.targetWidth
                } else {
                    availaibleSpace -= headerRow.targetWidth
                }

                var temp = availaibleSpace / minimalFilledArray.length
                var j = 0
                while (true) {
                    if (j >= minimalFilledArray.length) {
                        break
                    }

                    if (minimalFilledArray[j] > temp) {
                        availaibleSpace -= minimalFilledArray[j]
                        minimalFilledArray.splice(j, 1)

                        if (!minimalFilledArray.length) {
                            break
                        }

                        temp = availaibleSpace / minimalFilledArray.length
                        j = 0
                    } else {
                        j++
                    }
                }
                _availableSpace = headerRow.targetWidth
                appBarContainer._fillComponentsWidth = temp
            }
            return
        }
        appBarContainer._fillComponentsWidth = 0

        // No header
        if (!headerRow.visible) {
            if (spacersCount > 0) {
                _availableSpace = 0
                appBarContainer._spacersWidth = Math.max(availaibleSpace / spacersCount, 0)
            }
            return
        }

        // Was header, but now we haven't much space for it.
        if (headerRow.width > 0 && availaibleSpace < headerRow.targetWidth) {
            _availableSpace = Math.max(availaibleSpace, 0)
            appBarContainer._spacersWidth = 0
            return
        }

        // Was header, but now we have space more than need for header
        if (headerRow.width > 0 && availaibleSpace >= headerRow.targetWidth) {
            availaibleSpace -= headerRow.targetWidth
            _availableSpace = headerRow.targetWidth
            if (spacersCount > 0) {
                appBarContainer._spacersWidth = Math.max(availaibleSpace / spacersCount, 0)
            }
            return
        }

        // Wasn't header, but have space for it.
        if (headerRow.width <= 0 && availaibleSpace >= headerRow.targetWidth) {

            availaibleSpace -= Theme.horizontalPageMargin
                    + Theme.paddingMedium
                    - leftPadding
                    + headerRow.targetWidth

            _availableSpace = headerRow.targetWidth

            if (spacersCount > 0) {
                appBarContainer._spacersWidth = Math.max(availaibleSpace / spacersCount, 0)
            }
            return
        }

        // Try to fit it with proper paddings
        availaibleSpace -= Theme.horizontalPageMargin + Theme.paddingMedium - leftPadding

        if (availaibleSpace >= headerRow.targetWidth) {
            availaibleSpace += headerRow.anchors.leftMargin - headerRow.targetWidth
            _availableSpace = headerRow.targetWidth

            if (spacersCount > 0) {
                appBarContainer._spacersWidth = Math.max(availaibleSpace / spacersCount, 0)
            }
        } else {
            _availableSpace = Math.max(availaibleSpace, 0)
            appBarContainer._spacersWidth = 0
        }
    }

    function close() {
        _openStatus = 0
    }

    function open() {
        _openStatus = 1
    }

    function _resetSelectedItem() {
        if (_selectedItem && _selectedItem !== headerMouseArea) {
            _selectedItem._highlightedByAppBar = false
        }
        _selectedItem = undefined
    }


    function _triggerNgfHighlight() {
        // Avoid too many feedback calls at one time
        if (_ngfEffectHighlight && !_ngfEffectHighlightTimer.running) {
            _ngfEffectHighlightTimer.start()
            _ngfEffectHighlight.play()
        }
    }

    function _onProgressUpdate() {
        if (pulleyGesture.status === PulleyGestureArea.Idle
                || pulleyGesture.status === PulleyGestureArea.Activating) {
            return
        }

        if (pulleyGesture.activeIndex == -1) {
            cursor.hide()
            return
        }

        var currentIndex = 0

        const headerSelectable = headerClickable && headerRow.visible

        // Check is header selected by gesture
        if (headerSelectable) {
            if (pulleyGesture.activeIndex === 0) {
                if (_selectedItem === headerMouseArea) {
                    return
                }

                _triggerNgfHighlight()

                if (_selectedItem !== undefined) {
                    _selectedItem._highlightedByAppBar = false
                }

                _selectedItem = headerMouseArea

                cursor.move()
                return
            }
            currentIndex++
        }

        // Iterate over all interactive controls of appbar and find selected one
        for (var i in content) {
            const child = content[i]

            if (child.hasOwnProperty(__spacerSpecialProperty) || !child.visible || child.width === 0 || !child.enabled) {
                continue
            }

            if (pulleyGesture.activeIndex === currentIndex) {
                if (_selectedItem !== child) {
                    _triggerNgfHighlight()
                    if (_selectedItem !== undefined && _selectedItem !== headerMouseArea) {
                        _selectedItem._highlightedByAppBar = false
                    }
                    _selectedItem = child
                    _selectedItem._highlightedByAppBar = true
                    cursor.move()
                }
                return
            }
            currentIndex++
        }
        _resetSelectedItem()
        cursor.move()
    }

    // Stop all animation and start only target
    function _startAnimation(animation) {
        hideAnimation.stop()
        betweenSelectedAnimation.stop()
        fromSmallCursorAnimation.stop()
        cursorAnimationOff.stop()
        animation.start()
    }

    function _setStatusBar() {
        if (background) {
            Private.StatusBarWatcher.setStatusBar(backgroundRect.backgroundColor, 1, appBar)
        } else {
            Private.StatusBarWatcher.setStatusBar(backgroundRect.backgroundColor, 0, appBar)
        }
    }

    onWidthChanged: updateComponentsWidthTimer.restart()
    onBackgroundChanged: if (_active) _setStatusBar()
    Component.onDestruction: Private.StatusBarWatcher.restoreStatusBar(false, appBar)
    on_ActiveChanged: {
        if (_bottomPosition) {
            return
        }

        if (_active) {
            return _setStatusBar()
        }

        Private.StatusBarWatcher.restoreStatusBar(false, appBar)
    }

    anchors.top: _bottomPosition ? undefined : parent.top
    anchors.topMargin: {
        if (_bottomPosition) {
            return 0
        }

        const topMargin = SafeZoneRect.insets.top
        if (_landscape || __silica_applicationwindow_instance.displayMode === ApplicationDisplayMode.SafeZone) {
            return topMargin
        }

        const threshold = Theme.dp(20)

        if (SafeZoneRect._orientation & Orientation.Portrait) {
            if (Private.ScreenCorners.topLeft > threshold || Private.ScreenCorners.topRight > threshold) {
                return Math.max(topMargin, Private.ScreenCorners.topLeft * 0.5, Private.ScreenCorners.topRight * 0.5)
            }
            return topMargin
        }

        if (Private.ScreenCorners.bottomLeft > threshold || Private.ScreenCorners.bottomRight > threshold) {
            return Math.max(topMargin, Private.ScreenCorners.bottomLeft * 0.5, Private.ScreenCorners.bottomRight * 0.5)
        }
        return topMargin
    }
    anchors.bottom: _bottomPosition ? parent.bottom : undefined
    anchors.bottomMargin: _bottomPosition ? SafeZoneRect.insets.bottom : 0

    width: parent.width
    height: appBarBackground.height

    Item {
        anchors.fill: appBarBackground
        anchors.topMargin: -appBar.anchors.topMargin
        anchors.bottomMargin: -appBar.anchors.bottomMargin

        visible: opacity > 0
        opacity: appBarBackground.opacity

        Private.BackgroundRectangle {
            anchors.fill: parent

            visible: backgroundLoader.status !== Loader.Ready && backgroundRect.opacity < 1
            color: __silica_applicationwindow_instance._backgroundColor
        }

        Loader {
            id: backgroundLoader

            anchors.fill: parent

            sourceComponent: appBar._page && backgroundRect.opacity < 1 ? appBar._page.background : null
        }

        Rectangle {
            id: backgroundRect

            readonly property color backgroundColor: {
                if (appBar.monochrome) {
                    return Qt.tint(Theme.rgba(palette.primaryColor, 1.0),
                                   palette.colorScheme === Theme.DarkOnLight ? "#e6ffffff" : "#e6000000") // 90%
                }
                return Qt.tint(Theme.rgba(palette.highlightColor, 1.0),
                               palette.colorScheme === Theme.DarkOnLight ? "#ccffffff" : "#cc000000") // 80%
            }

            onBackgroundColorChanged: if (_active && background) _setStatusBar()

            anchors.fill: parent

            visible: opacity != 0
            opacity: appBar.background ? 1 : 0

            color: backgroundColor

            Behavior on opacity {NumberAnimation {duration: 200}}
            Behavior on color {ColorAnimation {duration: 200}}
        }

        // Do not propagate touch through AppBar background
        MouseArea {
            anchors.fill: parent
        }
    }

    z: Theme.ZOrderHeader // Make it over all components on page.
    visible: _bottomPosition // When _bottomPosition false visible would set to in onCompleted

    Item {
        id: appBarBackground

        width: parent.width
        height: {
            if (_splitViewhasMoreOneSplit) {
                return Theme._itemSizeHeader
            }
            return Theme._itemSizeHeader * _openStatus
        }
        opacity: {
            if (_splitViewhasMoreOneSplit) {
                return 1
            }

            return _openStatus
        }

        clip: true

        onOpacityChanged: {
            if (_pageActive && appBar.visible && opacity == 0) {
                Private.StatusBarWatcher.restoreStatusBar(true, appBar)
            }
        }

        Behavior on height { NumberAnimation { duration: 300 } }
        Behavior on opacity { FadeAnimation { duration: 300 } }

        Rectangle {
            y: _bottomPosition ? 0 : Theme._itemSizeHeader - height
            width: parent.width
            height: Theme._lineWidth
            visible: divider
            opacity: Theme.opacityFaint
            color: palette.primaryColor
        }

        Row {
            id: headerRow

            property real targetWidth: {
                var width = Math.max(headerColumnTargetWidth, iconContainerItem.width)
                if (iconContainerItem.visible && headerColumnTargetWidth != 0) {
                    width = iconContainerItem.width + spacing + headerColumnTargetWidth
                }

                if (appBar.maximumHeaderWidth) {
                    width = Math.min(width, appBar.maximumHeaderWidth)
                }

                if (visible && (headerClickable || appBar.minimumHeaderWidth)) {
                    width = Math.max(width, headerClickable ? Theme.iconSizeMedium : 0, appBar.minimumHeaderWidth)
                }

                return width
            }

            property real headerColumnTargetWidth: Math.max(headerLabel.contentWidth, subHeaderLabel.contentWidth, 0)

            onTargetWidthChanged: updateComponentsWidthTimer.restart()
            onWidthChanged: updateComponentsWidthTimer.restart()

            anchors {
                top: parent.top
                topMargin: (Theme._itemSizeHeader - headerRow.height) / 2
                left: parent.left
                leftMargin: Theme.horizontalPageMargin + _leftCutoutInset
            }

            width: {
                const minWidth = Math.max(headerClickable ? Theme.iconSizeMedium : 0, minimumHeaderWidth)
                return Math.max(Math.min(targetWidth,  _availableSpace), minWidth)
            }

            visible: !_bottomPosition && (iconContainerItem.shouldBeVisible || headerText != "" || subHeaderText != "")
            spacing: Theme.paddingMedium
            clip: true

            OpacityMask {
                id: iconContainerItem

                readonly property bool shouldBeVisible: iconContainerRectangle.color != "#00000000"
                                                        || imageItem.source != ""
                                                        || textItem.text != ""

                width: visible ? Theme.iconSizeMedium : 0
                height: Theme.iconSizeMedium
                visible: shouldBeVisible

                source: iconContainerRectangle

                maskSource: Rectangle {
                    objectName: "iconContainerMask"
                    width: Theme.iconSizeMedium
                    height: Theme.iconSizeMedium

                    visible: false
                }
            }

            Column {
                anchors.verticalCenter: parent.verticalCenter

                width: {
                    var width = parent.width
                    if (iconContainerItem.visible) {
                        width -= iconContainerItem.width + parent.spacing
                    }
                    return width
                }

                visible: (headerText != "" || subHeaderText != "") && width > 0

                Label {
                    id: headerLabel

                    width: parent.width
                    // Hack to look like figma does
                    lineHeight: 0.8
                    topPadding: -Theme.dp(7)
                    bottomPadding: Theme.dp(7)
                    visible: width > 0
                    truncationMode: TruncationMode.Fade
                    color: headerClickable ? palette.highlightColor: palette.primaryColor
                    font.pixelSize: Theme.fontSizeLarge
                    font.weight: Font.Medium
                    text: headerText
                }

                Label {
                    id: subHeaderLabel

                    width: parent.width
                    // Hack to look like figma does
                    lineHeight: 0.8
                    topPadding: -Theme.dp(4)
                    bottomPadding: Theme.dp(4)
                    visible: text != "" && width > 0
                    truncationMode: TruncationMode.Fade
                    color: headerClickable ? palette.secondaryHighlightColor : palette.secondaryColor
                    font.pixelSize: Theme.fontSizeExtraSmall
                    text: subHeaderText
                }
            }
        }


        MouseArea {
            id: headerMouseArea

            onClicked: appBar.headerClicked()

            anchors.centerIn: headerRow
            width: headerRow.width + 2 * Theme.paddingSmall
            height: Theme.dp(72)
            enabled: headerRow.visible && headerClickable

            Rectangle {
                anchors.fill: parent
                color: palette.highlightColor
                opacity: parent.containsMouse ? Theme.opacityFaint : 0
                radius: Theme.dp(6)
            }
        }

        Row {
            id: appBarContainer

            property real _spacersWidth
            property real _fillComponentsWidth

            function _updateComponentsWidth() {
                updateComponentsWidthTimer.restart()
            }

            function _setContext(text) {
                contextContainer.setContext(text)
            }

            function _resetContext() {
                contextContainer.reset()
            }

            anchors {
                onRightMarginChanged: _updateComponentsWidth()

                top: _bottomPosition ? undefined : parent.top
                topMargin: (Theme._itemSizeHeader - appBarContainer.height) * 0.5
                bottom: _bottomPosition ? parent.bottom : undefined
                bottomMargin: anchors.topMargin
                left: parent.left
                leftMargin: _cursorStartX
                right: parent.right
                rightMargin: {
                    if (_lastVisibleSearchField) {
                        return _rightCutoutInset + Theme.paddingLarge
                    }

                    if (_bottomPosition) {
                        return _rightCutoutInset + Theme.horizontalPageMargin
                    }

                    return Math.max(_rightCutoutInset + Theme.dp(4),
                                    appBar._landscape && !Screen._isTablet ? Theme.horizontalPageMargin : 0)
                }
            }

            height: Theme.dp(72)

            Component.onCompleted: {
                var haveSearchField = false

                for (var i in children) {
                    const obj = children[i]

                    if (obj.hasOwnProperty(__spacerSpecialProperty)) {
                        continue
                    }

                    // Bind verticalCenter - to centering all components inside appbar
                    obj.anchors.verticalCenter = Qt.binding(function() { return appBarContainer.verticalCenter })

                    if (obj.hasOwnProperty("containsMouse") && obj.hasOwnProperty("pressed")) {
                        backgroundComponent.createObject(obj)
                    }

                    if (obj.hasOwnProperty(__searchSpecialProperty)) {
                        if (haveSearchField) {
                            console.error("AppBar should contain only one AppBarSearchField.")
                        }

                        haveSearchField = true
                    }

                }
                updateComponentsWidthTimer.restart()
            }
        }

        // Not really smooth solution
        Component {
            id: backgroundComponent

            Rectangle {
                anchors.fill: parent
                color: palette.highlightColor
                opacity: Theme.opacityFaint
                radius: Theme.dp(6)
                visible: parent.containsMouse && parent.pressed
            }
        }

        Rectangle {
            id: cursor

            property real newX

            function hide() {
                if (hideAnimation.running || !cursor.visible) {
                    return
                }

                // Animate hiding when the cursor moves to another appbar
                newX = x
                if (pulleyGesture.mouseIsOverCancelZone) {
                    newX += width * 0.5
                } else if (pulleyGesture.actualSide() === PulleyGestureArea.Left) {
                    newX += width
                }
                _startAnimation(hideAnimation)

                _resetSelectedItem()
            }

            function move() {
                if (!_selectedItem) {
                    return
                }

                if (visible) {
                    // Cursor was visible
                    if (width == _cursorMinWidth) {
                        // Cursor was short
                        _startAnimation(fromSmallCursorAnimation)
                        return
                    }
                    _startAnimation(betweenSelectedAnimation)
                    return
                }
                // Cursor was invisible
                x = (_selectedItem === headerMouseArea ? 0 : _cursorStartX) + _selectedItem.x
                if (pulleyGesture.actualSide() === PulleyGestureArea.Right) {
                    // Start animation from right side
                    x += _selectedItem.width
                }

                width = _cursorMinWidth
                opacity = 0
                visible = true
                _startAnimation(fromSmallCursorAnimation)
            }

            anchors.top: parent.top
            anchors.topMargin: (Theme._itemSizeHeader - appBarContainer.height) * 0.5
            width: appBar._cursorMinWidth
            height: Theme.dp(72)
            color: palette.highlightColor
            opacity: Theme.opacityFaint
            radius: Theme.dp(6)
            visible: false

            Connections {
                target: _selectedItem ? _selectedItem : null

                onVisibleChanged: {
                    if (!cursorAnimationOff.running || !_selectedItem || _selectedItem.visible) {
                        return
                    }
                    cursorAnimationOff.stop()
                    cursor.visible = false
                    cursor.width = 0
                    cursor.x = 0
                    cursor.opacity = 0
                    _selectedItem = undefined
                }
            }

            SequentialAnimation {
                id: hideAnimation

                ParallelAnimation {
                    NumberAnimation {
                        target: cursor
                        duration: 100
                        properties: "width"
                        to: 0
                        easing.type: Easing.InOutQuad
                    }

                    NumberAnimation {
                        target: cursor
                        duration: 100
                        properties: "x"
                        to: cursor.newX
                        easing.type: Easing.InOutQuad
                    }

                    FadeAnimation {
                        target: cursor
                        duration: 100
                        to: 0
                    }
                }
                ScriptAction {
                    script: {
                        cursor.visible = false
                        cursor.opacity = Theme.opacityHigh
                        cursor.width = _cursorMinWidth
                    }
                }
            }

            ParallelAnimation {
                id: fromSmallCursorAnimation

                NumberAnimation {
                    target: cursor
                    duration: 70
                    properties: "width"
                    to: _selectedItem ? _selectedItem.width : 0
                    easing.type: Easing.InOutQuad
                }

                FadeAnimation {
                    target: cursor
                    duration: 75
                    to: Theme.opacityFaint
                }

                NumberAnimation {
                    target: cursor
                    properties: "x"
                    duration: 70
                    to: _selectedItem ? _selectedItem.x + (_selectedItem === headerMouseArea ? 0 : _cursorStartX) : 0
                    easing.type: Easing.InOutQuad
                }

            }

            ParallelAnimation {
                id: betweenSelectedAnimation

                NumberAnimation {
                    target: cursor
                    duration: 60
                    properties: "width"
                    to: _selectedItem ? _selectedItem.width : 0
                    easing.type: Easing.InOutQuad
                }

                FadeAnimation {
                    target: cursor
                    duration: 55
                    to: Theme.opacityFaint
                }

                NumberAnimation {
                    target: cursor
                    properties: "x"
                    duration: 60
                    to: _selectedItem ? _selectedItem.x + (_selectedItem === headerMouseArea ? 0 : _cursorStartX) : 0
                    easing.type: Easing.InOutQuad
                }
            }

            SequentialAnimation {
                id: cursorAnimationOff

                ScriptAction{
                    script: {
                        cursor.width = Qt.binding(function() {
                            if (_selectedItem){
                                return _selectedItem.width
                            }
                            return 0
                        })
                        cursor.x = Qt.binding(function() {
                            if (_selectedItem) {
                                return (_selectedItem === headerMouseArea ? 0 : _cursorStartX) + _selectedItem.x
                            }
                            return 0
                        })
                        if (_ngfEffectSelect) {
                            _ngfEffectSelect.play()
                        }
                        if (_selectedItem && _selectedItem !== headerMouseArea) {
                            _selectedItem._highlightedByAppBar = false
                        }
                    }
                }

                FadeAnimator {
                    target: cursor
                    duration: 55
                    from: Theme.opacityFaint
                    to: Theme.opacityHigh
                }

                FadeAnimator {
                    target: cursor
                    duration: 110
                    from: Theme.opacityHigh
                    to: Theme.opacityFaint
                }

                FadeAnimator {
                    target: cursor
                    duration: 55
                    from: Theme.opacityFaint
                    to: Theme.opacityHigh
                }

                FadeAnimator {
                    target: cursor
                    duration: 110
                    from: Theme.opacityHigh
                    to: Theme.opacityFaint
                }

                FadeAnimator {
                    target: cursor
                    duration: 55
                    from: Theme.opacityFaint
                    to: Theme.opacityHigh
                }

                FadeAnimator {
                    target: cursor
                    duration: 45
                    from: Theme.opacityHigh
                    to: 0
                }

                ScriptAction {
                    script: {
                        cursor.visible = false
                        cursor.width = 0
                        cursor.x = 0
                        cursor.opacity = 0
                        _selectedItem = undefined
                    }
                }
            }
        }
    }

    Item {
        id: contextContainer

        // This for long tap on button
        property string contextText
        property bool contextLongTap

        function setContext(text) {
            contextTimer.stop()
            contextText = text
            contextLongTap = true
        }

        function reset() {
            contextTimer.start()
        }

        anchors.horizontalCenter: parent.horizontalCenter
        y: Theme._itemSizeHeader + Theme.paddingSmall
        width: parent.width - Theme.dp(8)
        height: Theme.dp(72)
        visible: opacity > 0 && !_bottomPosition
        opacity: ((!cursorAnimationOff.running && _selectedItem && _selectedItem.context) || contextLongTap) && _openStatus == 1 ? 1 : 0

        Behavior on opacity { FadeAnimation {} }

        Rectangle {
            anchors.fill: parent

            color:Qt.tint(Qt.tint(palette.highlightColor,
                                  palette.colorScheme === Theme.DarkOnLight ? "#ccffffff" : "#cc000000"), // 80%
                          Theme.rgba(palette.highlightColor, Theme.opacityFaint))
            radius: Theme.dp(6)

            border.width: Theme._lineWidth
            border.color: Theme.rgba(palette.primaryColor, 0.1)
        }

        Label {
            anchors.centerIn: parent
            width: {
                // This if for make propper cutting of label with fading
                if (implicitWidth > parent.width - 2 * Theme.paddingMedium) {
                    return parent.width - 2 * Theme.paddingMedium
                }
                return implicitWidth
            }
            truncationMode: TruncationMode.Fade
            font.pixelSize: Theme.fontSizeExtraSmall
            color: palette.highlightColor
            text: _selectedItem && _selectedItem.context ? _selectedItem.context : contextContainer.contextText
        }

        Timer {
            id: contextTimer

            onTriggered: contextContainer.contextLongTap = false

            interval: 3000
        }
    }

    AppBarIconContainer {
        id: iconContainerObject

        radius: Theme.iconSizeMedium * 0.5
    }

    Rectangle {
        id: iconContainerRectangle
        objectName: "iconContainerRectangle"

        width: Theme.iconSizeMedium
        height: Theme.iconSizeMedium
        color: "transparent"
        visible: false

        Image {
            id: imageItem
            objectName: "iconContainerImageItem"

            anchors.centerIn: parent

            // Another hack, when primary color was changed - image provider also change icon color
            Connections {
                target: palette

                onPrimaryColorChanged: {
                    const source = imageItem.source.toString()

                    if (source != "" && (source.indexOf("image://") === 0
                                         || source.indexOf("?#") !== -1)) {
                        imageItem.source = ""
                        imageItem.source = source
                    }
                }
            }
        }

        Text {
            id: textItem
            objectName: "iconContainerTextItem"

            anchors.centerIn: parent
            width: implicitWidth > parent.width ? parent.width : implicitWidth
            font.pixelSize: Theme.fontSizeMedium
            font.family: Theme.fontFamily
            color: Theme.primaryColor
        }
    }

    Timer {
        id: _ngfEffectHighlightTimer

        interval: 30
    }

    Timer {
        id: updateComponentsWidthTimer

        onTriggered: _updateComponentsWidth()
        interval: 0
    }

    Connections {
        target: pulleyGesture.verticalGesture

        onProgressChanged: {
            if (_openStatus != 1) {
                _openStatus = pulleyGesture.verticalGesture.progress < 1
                        ? pulleyGesture.verticalGesture.progress * 0.5
                        : 1
            }
        }
    }

    PulleyGestureArea {
        id: pulleyGesture

        onStatusChanged: {
            switch (pulleyGesture.status) {
            case PulleyGestureArea.Idle:
                if (_openStatus < 1) {
                    _openStatus = 0
                }
                break
            case PulleyGestureArea.Activating:
                cursorAnimationOff.stop();
                cursor.visible = false
                _resetSelectedItem()
                break
            case PulleyGestureArea.GestureActive:
                _triggerNgfHighlight()
                _openStatus = 1
                break
            }
        }

        onActiveIndexChanged: _onProgressUpdate()

        onSelected: {
            if (!_selectedItem || pulleyGesture.activeIndex == -1) {
                cursor.visible = false

                return
            }

            _startAnimation(cursorAnimationOff)

            if (_selectedItem.hasOwnProperty("clicked")) {
                _selectedItem.clicked({
                    "accepted" : false,
                    "button" : Qt.LeftButton,
                    "buttons" : Qt.LeftButton,
                    "modifiers" : Qt.NoModifier,
                    "wasHeld" : false,
                    "x" : _selectedItem.width * 0.5,
                    "y" : _selectedItem.height * 0.5
                })
            }
        }

        onRejected: cursor.hide()

        onMouseIsOverCancelZoneChanged: {
            if (pulleyGesture.mouseIsOverCancelZone) {
                cursor.hide()
            } else {
                _onProgressUpdate()
            }
        }

        gestureEnabled: !_bottomPosition
                        && _pageActive
                        && _tabItemActive
                        && appBar.visible
                        && (content.length > 0 || _openStatus != 1)
                        && appBar.enabled
    }

    Component.onCompleted: {
        if (_bottomPosition) {
            return
        }

        if (Private.StatusBarWatcher.window != __silica_applicationwindow_instance) {
            Private.StatusBarWatcher.window = __silica_applicationwindow_instance
        }

        visible = true

        // avoid hard dependency to ngf module
        _ngfEffectSelect = Qt.createQmlObject("import org.nemomobile.ngf 1.0; NonGraphicalFeedback { event: 'pulldown_lock' }",
                                              appBar, 'NonGraphicalFeedback')
        _ngfEffectHighlight = Qt.createQmlObject("import org.nemomobile.ngf 1.0; NonGraphicalFeedback { event: 'pulldown_highlight' }",
                                                 appBar, 'NonGraphicalFeedback')
    }
}
