/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.6
import QtQuick.Layouts 1.1
import Sailfish.Silica 1.0

/*!
    \qmltype ActionsPlaceholder
    \inherits Item
    \inqmlmodule Aurora.Applications.Internal
    \brief Placeholder with any content and actions.
*/
Item {
    id: root

    /*! Is application in landscape orientation. */
    readonly property bool isLandscape: __silica_applicationwindow_instance.orientation &
                                        Orientation.LandscapeMask

    /*! Is large screen. */
    readonly property bool largeScreen: Screen.sizeCategory >= Screen.Large

    /*! Placeholder orientation. */
    property int orientation: root.isLandscape ? Qt.Horizontal : Qt.Vertical

    /*! Spacing between main and label with additional content section. */
    property real betweenSectionSpacing: Theme.dp(48)

    /*! Spacing between label and additional content. */
    property real insideSectionSpacing: Theme.dp(36)

    /*! Main content item. */
    property Item contentItem: null

    /*! User defined additional item. */
    property Item additionalContentItem: null

    /*! Alias to content item container. */
    property alias contentItemContainer: contentItemContainer

    /*! Alias to text label. */
    property alias label: label

    implicitWidth: rootLayout.implicitWidth
    implicitHeight: rootLayout.implicitHeight

    GridLayout {
        id: rootLayout

        anchors.centerIn: parent

        rowSpacing: root.betweenSectionSpacing
        columnSpacing: rootLayout.rowSpacing

        rows: 2
        columns: 1

        Item {
            id: contentItemContainer

            Layout.preferredWidth: root.contentItem
                                   ? root.contentItem.implicitWidth
                                   : 0
            Layout.preferredHeight: root.contentItem
                                    ? root.contentItem.implicitHeight
                                    : 0
            Layout.alignment: Qt.AlignHCenter

            data: root.contentItem
        }

        ColumnLayout {
            Layout.alignment: Qt.AlignHCenter

            spacing: 0

            Label {
                id: label

                Layout.preferredWidth: Math.min(implicitWidth + Theme.paddingMedium,
                                                root.width / (root.orientation === Qt.Horizontal
                                                              ? rootLayout.rows
                                                              : rootLayout.columns))
                Layout.alignment: Qt.AlignHCenter

                horizontalAlignment: Text.AlignHCenter
                elide: Text.ElideRight
                wrapMode: Text.Wrap
                maximumLineCount: 2
                font.pixelSize: Theme.fontSizeExtraLarge
            }

            Item {
                Layout.preferredHeight: root.insideSectionSpacing
                visible: root.additionalContentItem
            }

            Item {
                id: additionalContentItemContainer

                Layout.preferredWidth: root.additionalContentItem
                                       ? root.additionalContentItem.implicitWidth
                                       : 0
                Layout.preferredHeight: root.additionalContentItem
                                        ? root.additionalContentItem.implicitHeight
                                        : 0
                Layout.alignment: Qt.AlignHCenter

                data: root.additionalContentItem
            }
        }
    }

    states: [
        State {
            when: root.orientation === Qt.Horizontal

            PropertyChanges {
                target: rootLayout

                rows: 1
                columns: 2
            }
        }
    ]
}
