/* -*- indent-tabs-mode: nil; js-indent-level: 2 -*- */
/* vim: set ts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

const { FileUtils } = ChromeUtils.import("resource://gre/modules/FileUtils.jsm");
const { Downloads } = ChromeUtils.import("resource://gre/modules/Downloads.jsm");

Logger.debug("JSScript: PrintHelper.js loaded");

var gPrintHelper = null;

function debug(aMsg) {
  Logger.debug("PrintHelper.js:", aMsg);
}

function isPrintButton(aElement) {
  return aElement.tagName === "BUTTON" &&
         aElement.id === "print";
}

function launchDownloading(downloadOptions, self) {
  (async function() {
    let list = await Downloads.getList(Downloads.ALL);
    let download = await Downloads.createDownload(downloadOptions);

    download["idForPrinting"] = downloadOptions.target;
    sendAsyncMessage("embed:printpdfdownloadstarted", {idForPrinting: downloadOptions.target});
    download.start();
    list.add(download);

    download.onchange = function() {
      if (download.error) {
        sendAsyncMessage("embed:printpdfdownloadfinished", {});
        sendAsyncMessage("embed:printpdfdownloaderror", {});
        downloadList.remove(download);
        return;
      }

      if (download.succeeded) {
        sendAsyncMessage("embed:printpdfdownloadfinished", {});
        self._createPrintDialog();
        downloadList.remove(download);
      }

      if (download.canceled) {
        downloadList.remove(download);
      }
    }
  })().then(null, Cu.reportError);
}

function cancelDownloading(idForPrinting) {
  (async function() {
    let downloadList = await Downloads.getList(Downloads.ALL);
    let list = await downloadList.getAll();
    for (let download of list) {
      if (download.idForPrinting === idForPrinting) {
        download.cancel();
        sendAsyncMessage("embed:printpdfdownloadfinished", {});
        break;
      }
    }
  })().then(null, Cu.reportError);
}

function PrintDialog(aData) {
  this._init(aData)
}

PrintDialog.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver]),

  _init: function(aData) {
    addMessageListener("embedui:printresponse", this);
    sendAsyncMessage("embed:print", aData);
  },

  receiveMessage: function(aMessage) {
    removeMessageListener("embedui:printresponse", this);
    this.onDone();
  },

  onDone: function(aResult) {}
};

function PrintHelper() {
  this._init();
}

PrintHelper.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver,
                                          Ci.nsISupportsWeakReference]),

  _dialog: null, // a proxy for modal subwindow
  _fileName: "",

  _init: function() {
    addEventListener("click", this, false);
    addMessageListener("embedui:print", this);
    addMessageListener("embedui:canceldownloading", this);
  },

  handleEvent: function(aEvent) {
    switch (aEvent.type) {
      case "click":
      this.onClicked(aEvent);
      break;
    }
  },

  receiveMessage: function receiveMessage(aMessage) {
    switch (aMessage.name) {
      case "embedui:print": {
        // We recieve from qml already unique file and correct file
        this._fileName = aMessage.data.to;

        var downloadOptions = {
          source: Services.ww.activeWindow,
          target: aMessage.data.to,
          saver: "pdf",
          contentType: "application/pdf"
        };

        let self = this;
        launchDownloading(downloadOptions, self);
        break;
      }
      case "embedui:canceldownloading": {
        cancelDownloading(aMessage.data.idForPrinting);
        break;
      }
    }
  },

  reset: function() {
    this._dialog = null;
  },

  onClicked: function(aEvent) {
    if (!isPrintButton(aEvent.target)) {
      return;
    }

    let randomStr = (Math.random() + 1).toString(36).substring(2);
    let fileName = "/tmp/" + randomStr + ".pdf";
    this._fileName = fileName;

    var downloadOptions = {
        source: {url: content.window.location.href, private: false},
        target: fileName,
    };

    let self = this;
    launchDownloading(downloadOptions, self);
  },

  _createPrintDialog: function _createPrintDialog() {
    let that = this;

    this._dialog = new PrintDialog({
      "filePath": this._fileName
    });

    this._dialog.onDone = function (result) {
      that.reset();
    }
  }
};

gPrintHelper = new PrintHelper();
 