/****************************************************************************
 **
 ** SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 ** SPDX-License-Identifier: Proprietary
 **
 ****************************************************************************/

import QtQuick 2.2
import QtQuick.Window 2.1 as QtQuick
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private

Private.SlideableBase {
    id: container

    property Item _pendingCurrentItem
    property real switchThreshold: QtQuick.Screen.pixelDensity * 10 // 10mm
    property real switchDistance: silicaConfiguration.switchDistance
    property real switchScaleBound: silicaConfiguration.switchScaleBound
    property real switchFixupDuration: silicaConfiguration.switchFixupDuration
    property real switchFixupMinimalDuration: silicaConfiguration.switchFixupMinimalDuration
    property real switchCurrentFactor: silicaConfiguration.switchCurrentFactor
    property real switchAlternateFactor: silicaConfiguration.switchAlternateFactor

    property bool pan: true
    property bool panning
    readonly property bool moving: panning || _animating
    property real absoluteProgress
    property real smoothedProgress

    // Pannable compatibilty
    property alias dragArea: dragArea
    property alias pannableItems: content.children
    property alias pannableParent: content

    property alias contentItem: content

    readonly property bool _vertical: (flow % 180) != 0
    readonly property bool _inverted: flow >= 180
    property real progress
    property real _currentPos
    property real _minimumPos
    property real _maximumPos

    property bool _animating
    property bool _switched

    signal movementEnded()

    property real overshoot: {
        if (!currentItem) {
            return 0
        } else if ((currentItem.Private.Slide.isLast && !_inverted)
                || (currentItem.Private.Slide.isFirst && _inverted)) {
            return -Math.min(0, absoluteProgress)
        } else if ((currentItem.Private.Slide.isLast && _inverted)
                || (currentItem.Private.Slide.isFirst && !_inverted)) {
            return Math.max(0, absoluteProgress)
        } else {
            return 0
        }
    }
    Behavior on overshoot {
        SmoothedAnimation {
            duration: 200
            easing.type: Easing.InOutQuad
        }
    }

    property int _flickDirection: panning && _currentPos != 0
            ? (_currentPos < 0 ? Private.Slide.Forward : Private.Slide.Backward)
            : 0
    on_FlickDirectionChanged: {
        if (_flickDirection !== Private.Slide.NoDirection) {
            _anchorAlternate()
        }
    }

    onPanningChanged: {
        _switched = false
        if (panning) {
            _stopAnimation()
        } else {
            if (_alternateItem && Math.abs(absoluteProgress) >= switchThreshold) {
                const prevCurrentItem = currentItem
                _switched = true
                _pendingCurrentItem = null
                currentItem = _alternateItem
                _anchorAlternate(prevCurrentItem)
                currentItem.visible = true
            }

            _startAnimation(Easing.OutQuad)
        }
    }

    Behavior on smoothedProgress {
        id: smoothedProgressBehavior

        enabled: false

        SmoothedAnimation {
            duration: switchFixupDuration
            velocity: 1000 / duration
            reversingMode: SmoothedAnimation.Immediate
            easing.type: Easing.OutQuad
        }
    }

    onAbsoluteProgressChanged: {
        if (panning && !smoothedProgressBehavior.enabled) {
            // reset smoothed value before write the first absoluteProgress
            smoothedProgress = 0
            smoothedProgressBehavior.enabled = true
        }

        smoothedProgress = Math.max(_minimumPos, Math.min(absoluteProgress, _maximumPos))

        if (!panning) {
            // disable smoothing after saving the last absoluteProgress
            smoothedProgressBehavior.enabled = false
        }
    }

    onSmoothedProgressChanged: {
        if (panning) {
            if (_vertical) {
                currentItem.y = smoothedProgress
            } else {
                currentItem.x = smoothedProgress
            }
        }
    }

    onProgressChanged: {
        currentItem.Private.Slide._switchProgress = panning || !_switched
                                                  ? 1 - progress * switchCurrentFactor
                                                  : (1 - progress) * 3 - switchAlternateFactor

        currentItem.opacity = currentItem.Private.Slide.switchProgress
        currentItem.scale = 1 - switchScaleBound + currentItem.opacity * switchScaleBound

        if (_alternateItem && _alternateItem != currentItem) {
            _alternateItem.Private.Slide._switchProgress = panning || !_switched
                                                         ? progress * 3 - switchAlternateFactor
                                                         : 1 - (1 - progress) * switchCurrentFactor

            _alternateItem.opacity = _alternateItem.Private.Slide.switchProgress
            _alternateItem.scale = 1 - switchScaleBound + _alternateItem.opacity * switchScaleBound
        }
    }

    function _startAnimation(easing) {
        const value = _vertical ? currentItem.y : currentItem.x

        currentItemAnimation.easing.type = easing

        alternateItemAnimation.to =
            value > 0 || !_inverted && currentItem.Private.Slide.isLast && value == 0
            ? -switchDistance : switchDistance

        alternateItemAnimation.from = _alternateItem
            ? (_vertical ? _alternateItem.y : _alternateItem.x)
            : alternateItemAnimation.to

        fixupAnimation.restart()
    }

    function _stopAnimation() {
        fixupAnimation.stop()
    }

    function _animationStopped() {
        if (_alternateItem) {
            _alternateItem.visible = false
            _alternateItem.opacity = 0
            _alternateItem.scale = 1
            _alternateItem.Private.Slide._switchProgress = 0
            _clearAnchors(_alternateItem)
            _alternateItem.Private.Slide.cleanup()
            _alternateItem = null
        }
        movementEnded()
        if (_pendingCurrentItem) {
            setCurrentItem(_pendingCurrentItem, true)
        } else if (currentItem) {
            currentItem.focus = true
            currentItem.opacity = 1
            currentItem.scale = 1
            currentItem.Private.Slide._switchProgress = 1
            absoluteProgress = currentItem.y
            updateSlidesTimer.restart()
        }
    }

    function _anchorAlternate(prevCurrentItem) {
        if (_alternateItem) {
            _clearAnchors(_alternateItem)
            _alternateItem.visible = _alternateItem === currentItem
                    || (currentItem.Private.Slide.forward === currentItem.Private.Slide.backward && _currentPos != 0)
            _alternateItem = null
        }
        if (_currentPos < 0) {
            if (!currentItem.Private.Slide.forward && !currentItem.Private.Slide.isLast) {
                createAdjacentItem(currentItem, Private.Slide.Forward)
            }

            if (currentItem.Private.Slide.forward) {
                _alternateItem = currentItem.Private.Slide.forward
                _anchorToForwardSide(_alternateItem.anchors, currentItem)
                _alternateItem.visible = true
            }
        } else if (_currentPos > 0) {
            if (!currentItem.Private.Slide.backward && !currentItem.Private.Slide.isFirst) {
                createAdjacentItem(currentItem, Private.Slide.Backward)
            }

            if (currentItem.Private.Slide.backward){
                _alternateItem = currentItem.Private.Slide.backward
                _anchorToBackwardSide(_alternateItem.anchors, currentItem)
                _alternateItem.visible = true
            }
        } else if (prevCurrentItem) {
            _alternateItem = prevCurrentItem
        }
    }

    function setCurrentItem(item, animate, direction) {
        if (panning) {
            return
        }
        _pendingCurrentItem = null
        if (!animate || !currentItem) {
            var previousItem = currentItem

            _stopAnimation()
            if (_alternateItem) {
                _clearAnchors(_alternateItem)
                _alternateItem = null
            }
            if (currentItem) {
                currentItem.visible = currentItem === item
            }
            _switched = true
            currentItem = item
            currentItem.x = 0
            currentItem.y = 0
            currentItem.visible = true
            currentItem.scale = 1
            currentItem.opacity = 1
            currentItem.focus = true
            currentItem.Private.Slide._switchProgress = 1

            if (previousItem && previousItem !== currentItem) {
                previousItem.Private.Slide.cleanup()
            }
            updateSlidesTimer.restart()
        } else if (item === currentItem) {
            // It's on its way.
        } else if (_animating) {
            _pendingCurrentItem = item
        } else {
            if (direction === Private.Slide.Forward || direction === Private.Slide.Backward) {
                // Respect the supplied value.
            } else if (direction === "left") { // Pannable compatibility
                direction = Private.Slide.Backward
            } else if (direction === "right") {
                direction = Private.Slide.Forward
            } else if (currentItem.Private.Slide.backward === item) {
                direction = Private.Slide.Forward
            } else if (currentItem.Private.Slide.forward === item) {
                direction = Private.Slide.Backward
            }

            // Establish the position the new current item will animate from by anchoring it to the
            // outgoing current item.
            if (direction === Private.Slide.Backward) {
                _anchorToForwardSide(item.anchors, currentItem)
            } else {
                _anchorToBackwardSide(item.anchors, currentItem)
            }
            _clearAnchors(item)

            _alternateItem = currentItem
            currentItem = item
            if (direction === Private.Slide.Backward) {
                _anchorToBackwardSide(_alternateItem.anchors, currentItem)
            } else {
                _anchorToForwardSide(_alternateItem.anchors, currentItem)
            }

            currentItem.visible = true
            _startAnimation(Easing.InOutQuad)
        }
    }

    function _clearAnchors(item) {
        item.anchors.left = undefined
        item.anchors.top = undefined
        item.anchors.right = undefined
        item.anchors.bottom = undefined
        item.anchors.margins = 0
    }

    function _anchorToBackwardSide(anchors, item) {
        switch (flow) {
        case Private.Slide.RightToLeft:
            anchors.leftMargin = switchDistance
            anchors.left = item.left
            anchors.top = item.top
            break;
        case Private.Slide.TopToBottom:
            anchors.bottomMargin = switchDistance
            anchors.left = item.left
            anchors.bottom = item.bottom
            break;
        case Private.Slide.BottomToTop:
            anchors.bottomMargin = switchDistance
            anchors.left = item.left
            anchors.top = item.top
            break;
        default:
            anchors.rightMargin = switchDistance
            anchors.top = item.top
            anchors.right = item.right
        }
    }

    function _anchorToForwardSide(anchors, item) {
        switch (flow) {
        case Private.Slide.RightToLeft:
            anchors.rightMargin = switchDistance
            anchors.top = item.top
            anchors.right = item.right
            break;
        case Private.Slide.TopToBottom:
            anchors.topMargin = switchDistance
            anchors.left = item.left
            anchors.top = item.top
            break;
        case Private.Slide.BottomToTop:
            anchors.bottomMargin = switchDistance
            anchors.left = item.left
            anchors.bottom = item.bottom
            break;
        default:
            anchors.leftMargin = switchDistance
            anchors.left = item.left
            anchors.top = item.top
        }
    }

    function _adjacentHeight(item) {
        return item && item.height < height ? item.height : height
    }

    function _updateSlides() {
        // move the slides to the expected position and update the offset value
        if (!currentItem) {
            return
        }

        var alternate = currentItem.Private.Slide.forward
        if (alternate) {
            _anchorToForwardSide(alternate.anchors, currentItem)
            _clearAnchors(alternate)
            alternate.Private.Slide._offset = _vertical ? alternate.y : alternate.x
        }

        alternate = currentItem.Private.Slide.backward
        if (alternate) {
            _anchorToBackwardSide(alternate.anchors, currentItem)
            _clearAnchors(alternate)
            alternate.Private.Slide._offset = _vertical ? alternate.y : alternate.x
        }
    }

    ParallelAnimation {
        id: fixupAnimation

        property real value: Math.abs(container._vertical ? container.currentItem.y : container.currentItem.x)
                             / container.switchDistance

        running: false

        onStarted: container._animating = true
        onStopped: {
            container._animating = false
            container._animationStopped()
        }

        NumberAnimation {
            id: currentItemAnimation

            target: container.currentItem
            property: container._vertical ? "y" : "x"
            duration: container.currentItem
                      ? Math.max(container.switchFixupMinimalDuration,
                                 container.switchFixupDuration * fixupAnimation.value)
                      : container.switchFixupMinimalDuration
            to: 0
            easing.type: Easing.OutQuad
        }
        NumberAnimation {
            id: alternateItemAnimation

            target: container.alternateItem
            property: _vertical ? "y" : "x"
            duration: container.alternateItem
                      ? Math.max(container.switchFixupMinimalDuration,
                                 container.switchFixupDuration * fixupAnimation.value)
                      : container.switchFixupMinimalDuration
            to: 0
            easing.type: currentItemAnimation.easing.type
        }
    }

    MouseArea {
        id: dragArea

        anchors.fill: parent
        objectName: "Pannable_dragArea"

        drag {
            target: enabled ? dragTarget : null
            filterChildren: true
            axis: container._vertical ? Drag.YAxis : Drag.XAxis
            minimumX: container._minimumPos
            maximumX: container._maximumPos
            minimumY: container._minimumPos
            maximumY: container._maximumPos
            threshold: QtQuick.Screen.pixelDensity * 5 // 5mm
        }

        states: [
            State {
                name: "drag-horizontal"
                when: dragArea.drag.active && !container._vertical
                PropertyChanges {
                    target: container
                    absoluteProgress: dragTarget.x
                }
            }, State {
                name: "drag-vertical"
                when: dragArea.drag.active && container._vertical
                PropertyChanges {
                    target: container
                    absoluteProgress: dragTarget.y
                }
            }
        ]

        transitions: [
            Transition {
                to: ""
                SequentialAnimation {
                    PropertyAction { target: container; property: "panning"; value: false }
                    PropertyAction { target: container; property: "absoluteProgress" }
                    PropertyAction { target: dragTarget; property: "x"; value: 0 }
                    PropertyAction { target: dragTarget; property: "y"; value: 0 }
                }
            }, Transition {
                from: ""
                SequentialAnimation {
                    PropertyAction { target: container; property: "absoluteProgress" }
                    PropertyAction { target: container; property: "panning"; value: container.pan }
                }
            }
        ]

        MouseArea {
            id: content

            onChildrenChanged: updateSlidesTimer.restart()

            anchors.fill: parent
            objectName: "Pannable_dragAreaContentItem"
        }

        Item {
            id: dragTarget

            states: [
                State {
                    when: !container.currentItem
                }, State {
                    name: "left-to-right"
                    when: container.flow === Private.Slide.LeftToRight
                    PropertyChanges {
                        target: container
                        _currentPos: container.currentItem.Private.Slide.offset
                        _minimumPos: container.currentItem.Private.Slide.isLast
                                     ? 0
                                     : -switchDistance
                        _maximumPos: container.currentItem.Private.Slide.isFirst
                                     ? 0
                                     : switchDistance
                        progress: Math.abs(container._currentPos / switchDistance)
                    }
                }, State {
                    name: "right-to-left"
                    when: container.flow === Private.Slide.RightToLeft
                    PropertyChanges {
                        target: container
                        _currentPos: -container.currentItem.Private.Slide.offset
                        _minimumPos: container.currentItem.Private.Slide.isFirst
                                     ? 0
                                     : -switchDistance
                        _maximumPos: container.currentItem.Private.Slide.isLast
                                     ? 0
                                     : switchDistance
                        progress: Math.abs(container._currentPos / switchDistance)
                    }
                }, State {
                    name: "top-to-bottom"
                    when: container.flow === Private.Slide.TopToBottom
                    extend: "left-to-right"
                }, State {
                    name: "bottom-to-top"
                    when: container.flow === Private.Slide.BottomToTop
                    extend: "right-to-left"
                }
            ]
        }
    }

    Timer {
        id: updateSlidesTimer

        onTriggered: {
            var child
            for (var i = 0; i < content.children.length; i++) {
                child = content.children[i]
                if (child === currentItem) {
                    child.opacity = 1
                    child.Private.Slide._switchProgress = 1
                } else {
                    child.opacity = 0
                    child.Private.Slide._switchProgress = 0
                }
            }

            _updateSlides()
        }

        interval: 1
    }

    Private.SilicaConfiguration { id: silicaConfiguration }

    Component.onCompleted: updateSlidesTimer.restart()
}
