/**
* SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
* SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.amberpdf 1.0
import ru.omp.print 1.0
import Aurora.Controls 1.0

Dialog {
    id: preview

    property string printPath
    property string currentPageRange
    property PrinterSettings printerSettings
    property bool centerPagesInLandscapeOrientation

    signal updatePageRange(string pageRange)

    PdfDocument {
        id: pdfiumProvider
    }

    AppBar {
        id: headerBar

        AppBarSpacer {}

        AppBarButton {
            id: componentButton

            onClicked: accept()

            //% "Print"
            text: qsTrId("components_pickers-he-print")

            icon.source: "image://theme/icon-splus-print"
        }
    }


    Component {
        id: switcherComponent

        Switch {
            id: paperSwitcher

            onClicked: state = "selected"

            anchors.right: parent === null ? undefined : parent.right

            automaticCheck: true
            highlighted: true

            checked: parent && parent.checked
            states: [
                State {
                    name: "selected"
                    PropertyChanges {
                        target: paperSwitcher.parent;
                        opacity: paperSwitcher.checked ? 1.0 : Theme.opacityLow;
                        checked: paperSwitcher.checked
                     }
                }
            ]
        }
    }

    PdfView {
        id: pdfView

        onSpecialPagesCountChanged: {
            countInfo.text = " | %1".arg(pdfView.specialPagesCount < 0 ? 0 : pdfView.specialPagesCount)
            pageInput.validator.top = pdfView.specialPagesCount
        }

        onPageRangeChanged: {
            currentPageRange = pdfView.pageRange
            showAllButton.enabled = currentPageRange.length != 0
            updatePageRange(currentPageRange)
        }

        onMaxPageIndexChanged: {
            pageInput.validator.bottom = pdfView.minPageIndex + 1
            pageInput.validator.top = pdfView.maxPageIndex + 1
        }

        onMinPageIndexChanged: {
            pageInput.validator.bottom = pdfView.minPageIndex + 1
            pageInput.validator.top = pdfView.maxPageIndex + 1
        }

        x: Theme.horizontalPageMargin
        width: parent.width - Theme.horizontalPageMargin * 2
        clip: true
        pdfPath: printPath
        pageRange: currentPageRange
        pageSpacing: Theme.paddingLarge + Theme.paddingSmall + Theme.fontSizeTiny
        documentProvider: pdfiumProvider
        reverse: printerSettings.pageOrder == PrinterSettings.Reverse
        grayScaleRendering: printerSettings.colorMode == PrinterSettings.GrayScale
        hiddenPagesOpacity: Theme.opacityLow
        pageCheckComponent: pageRange.length == 0 ? switcherComponent : null

        pageNumberComponent: Component {
            Item {
                property string currentPageNumber: ""
                property real bottomMargin: Theme.paddingSmall + Theme.fontSizeTiny / 2
                Label {
                    anchors.verticalCenter: parent.verticalCenter

                    text: currentPageNumber
                    color: Theme.secondaryColor
                    font.pixelSize: Theme.fontSizeTiny
                }
            }
        }

        Component.onCompleted: {
            if (printerSettings.format && printerSettings.format.width > 0 && printerSettings.format.height > 0) {
                if (printerSettings.orientation == PrinterSettings.Portrait) {
                    pageAspectRatio = printerSettings.format.width / printerSettings.format.height
                    centering = false
                } else if (printerSettings.orientation == PrinterSettings.Landscape) {
                    pageAspectRatio = printerSettings.format.height / printerSettings.format.width
                    fitToPage = printerSettings.scaleToPaper
                    centering = centerPagesInLandscapeOrientation
                }
            }
            y = headerBar.height
            height = root.height - headerBar.height
        }
    }

    DockedPanel {
        id: toolBar

        anchors.leftMargin: Theme.paddingLarge

        width: parent.width
        height: Theme.itemSizeLarge

        open: true
        dock: Dock.Bottom

        background: Rectangle {
            anchors.fill: parent

            color: Qt.tint(Theme.rgba(palette.highlightColor, 1.0),
                   palette.colorScheme === Theme.DarkOnLight ? "#ccffffff" : "#cc000000")
        }

        Button {
            id: showAllButton

            onClicked: {
                showAllButton.forceActiveFocus()
                popupMenu.open(showAllButton)
            }

            anchors {
                verticalCenter: pageControl.verticalCenter
                left: parent.left
                leftMargin: Theme.paddingLarge
            }

            width: Theme.dp(72)
            height: Theme.dp(72)

            backgroundColor: "transparent"
            highlighted: activeFocus
            enabled: currentPageRange.length != 0

            icon.source: "image://theme/icon-splus-more"

            PopupMenu {
                id: popupMenu

                x: Theme.paddingLarge
                width: toolBar.width - Theme.paddingLarge * 2
                headerBackground: PopupMenuHeader.SolidBackground

                PopupMenuCheckableItem {
                    onCheckedChanged: pdfView.pageCheckComponent = checked ? switcherComponent : null

                    width: parent.width - Theme.paddingLarge
                    height: Theme.itemSizeSmall

                    //% "Display the whole document"
                    text: qsTrId("components_pickers-la-display-the-whole-document")

                    //% "Show unprintable pages"
                    hint: qsTrId("components_pickers-la-unprintable-pages")

                    checked: pdfView.pageCheckComponent != null

                    closeOnClick: true
                }
            }
        }

        SecondaryButton {
            id: pageControl

            onClicked: pageInput.forceActiveFocus()

            anchors.horizontalCenter: parent.horizontalCenter

            width: (root.width - Theme.horizontalPageMargin * 2) / 3
            height: Theme.itemSizeMedium

            backgroundColor: "transparent"
            border.color: Theme.rgba(Theme.primaryColor, Theme.opacityFaint)

            Row {
                anchors.centerIn: parent

                TextInput  {
                    id: pageInput

                    onAccepted: {
                        pdfView.goToPage(parseInt(text) - 1)
                        focus = false
                    }
                    onActiveFocusChanged: {
                        countInfo.visible = !activeFocus
                        color: activeFocus ? Theme.highlightColor : Theme.primaryColor
                    }

                    anchors.verticalCenter: parent.verticalCenter
                    color: activeFocus ? Theme.highlightColor : Theme.primaryColor
                    text: "%1".arg(pdfView.currentIndex <= 0 ? 1 : pdfView.currentIndex + 1)

                    font {
                        pixelSize: Theme.fontSizeSmall
                        weight: Font.DemiBold
                    }

                    activeFocusOnPress: true
                    inputMethodHints: Qt.ImhFormattedNumbersOnly

                    validator: IntValidator{top: pdfView.maxPageIndex + 1; bottom: pdfView.minPageIndex + 1;}
                }
                Label {
                    id: countInfo

                    anchors.verticalCenter: parent.verticalCenter

                    text: " | %1".arg(pdfView.specialPagesCount < 0 ? 0 : pdfView.specialPagesCount)
                    font.pixelSize: Theme.fontSizeSmall
                    font.weight: Font.DemiBold
                    color: Theme.secondaryColor
                }
            }
        }
    }
}
