/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

pragma Singleton

import QtQuick 2.6
import Sailfish.Silica 1.0

/*!
    \qmltype UiUtils
    \inherits QtObject
    \inqmlmodule Aurora.Applications.Internal
    \brief Ui utils singleton.
*/
QtObject {
    id: root

    /*! Default Aurora::PopupMenuItem height. */
    readonly property real popupMenuItemHeight: Theme.dp(72)

    /*! Default Aurora::PopupMenu preferredWidth. */
    readonly property real popupMenuPreferredWidth:
        Math.max(Screen.width, Screen.height) * 0.33333333 - Theme.horizontalPageMargin

    /*! Is this phone or tablet. */
    readonly property bool isLargeScreen: Screen.sizeCategory >= Screen.Large

    /*! \internal QtObject cannot have children. Use this property as child container. */
    readonly property Item _d: Item {
        TextMetrics {
            id: popupMenuItemTextMetrics

            font.pixelSize: Theme.fontSizeMedium // from PopupMenuBaseItem
        }

        TextMetrics {
            id: popupMenuItemHintMetrics

            font.pixelSize: Theme.fontSizeExtraSmall // from PopupMenuBaseItem
        }
    }

    /*!
      \brief Get path to theme icon.
      \a args - object with next interface:
                {
                    name: string,
                    color?: string,
                }
      */
    function getIcon(args) {
        if (!args) {
            console.warn("No arguments passed")
            return null
        }

        if (!args.name) {
            console.warn("No name passed")
            return null
        }

        var path = "image://theme/%1".arg(args.name)

        if (args.color) {
            path += "?%1".arg(args.color)
        }

        return path
    }

    /*!
      \brief Get Aurora::PopupMenu::preferredWidth by its largest item.
      \a args - object with next interface:
                {
                    popupMenu: Aurora::PopupMenu,
                    minimumWidth?: number,
                    maximumWidth?: number
                }
      */
    function getPopupMenuPreferredWidth(args) {
        if (!args.popupMenu) {
            console.warn("No popupMenu passed")
            return 0
        }

        var includeIcons = false

        var preferredWidth = Math.max.apply(
            Math,
            Array.apply(null, Array(args.popupMenu.items.length)).map(function(_, i) {
                const item = args.popupMenu.items[i]

                /* If we use for example Repeater inside
                   PopupMenu there is no text property. */
                if (!item.text || !item.icon) {
                    return 0
                }

                popupMenuItemTextMetrics.text = item.text
                popupMenuItemHintMetrics.text = item.hint

                if (!includeIcons && item.icon.source.toString()) {
                    includeIcons = true
                }

                return Math.max(popupMenuItemTextMetrics.width,
                                popupMenuItemHintMetrics.width)
            })
        ) + Theme.paddingSmall * 2 // PopupMenuBaseItem internal Row {} paddings
          + Theme.paddingSmall * 4 // PopupMenuBaseItem internal paddigns (left && right)
          + (includeIcons ? root.popupMenuItemHeight // PopupMenuBaseItem icon container size
                          : Theme.dp(18) - Theme.paddingSmall)

        if (args.minimumWidth != null) {
            preferredWidth = Math.max(preferredWidth, args.minimumWidth)
        }

        if (args.maximumWidth != null) {
            preferredWidth = Math.min(preferredWidth, args.maximumWidth)
        }

        return preferredWidth
    }

    /*!
      Get formatted string from date \a source: date | string.
      \a literals - object of string literals.
      */
    function getDateSection(source, literals) {
        if (!literals) {
            literals = {
                today: "",
                yesterday: ""
            }
        }

        const date = (function() {
            var temp = new Date(source)

            if (isNaN(temp)) {
                temp = source.split("-")
                return new Date(temp[0], temp[1] - 1, temp[2])
            }

            return temp
        })()

        const today = new Date()

        if (root.isDateEquals(date, today)) {
            return literals.today
        }

        const yesterday = (function() {
            const temp = new Date(today.getTime())
            temp.setDate(temp.getDate() - 1)
            return temp
        })()

        if (root.isDateEquals(date, yesterday)) {
            return literals.yesterday
        }

        const dateText = today.getFullYear() - date.getFullYear() >= 1
                       ? Format.formatDate(source, Formatter.DateFull)
                       : Format.formatDate(source, Formatter.DateFullWithoutYear)

        const firstLetter = dateText[0]

        return firstLetter.toUpperCase() + dateText.slice(1)
    }

    /*!
      Compare date \a d0 with date \a d1 without time difference.
      */
    function isDateEquals(d0, d1) {
        return d0.getFullYear() === d1.getFullYear() &&
                d0.getMonth() === d1.getMonth() &&
                d0.getDate() === d1.getDate()
    }
}
