/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.6

/*!
    \qmltype LazyLoader
    \inherits Loader
    \inqmlmodule Aurora.Applications.Internal
    \brief Lazy loader.
*/
Loader {
    id: root

    /*!
      Instance of created item.
      */
    readonly property Item instance: root.item

    /*!
      Function will call callback
        \a args.resolve if item created
        \a args.reject if item cannot be created
      */
    function promise(args) {
        d.args = d.initializeArgs(args)

        if (d.getResolve()) {
            d.resolve(args)
            return
        }

        root.active = true
    }

    /*!
      Force recreate internal item with next promise call.
      Will be called on source or sourceComponent changed.
      */
    function markAsDirty() {
        d.recreate = true
    }

    active: false

    asynchronous: true

    onStatusChanged: {
        if (root.status === Loader.Error) {
            d.reject()
            return
        }

        if (root.status === Loader.Ready && d.recreate && root.item) {
            d.resolve(d.args)

            d.recreate = false
        }
    }

    onSourceChanged: root.markAsDirty()
    onSourceComponentChanged: root.markAsDirty()

    QtObject {
        id: d

        property var args: null

        property bool recreate: true

        function initializeArgs(args) {
            args = args || {}

            args.resolve = args.resolve || function() {
                console.warn("No resolve callback.")
            }

            args.reject = args.reject || function() {
                console.warn("No reject callback.")
            }

            return args
        }

        function resolve(args) {
            args.resolve(root.item, getActionOutputArgs())
        }

        function reject(args) {
            console.warn("Error with creation.")

            args.reject()
        }

        function getResolve() {
            return !!root.item && !d.recreate
        }

        function getActionOutputArgs() {
            return {
                instance: root.item,
                loader: root
            }
        }
    }
}
