/****************************************************************************
**
** Copyright (c) 2020-2023 Open Mobile Platform LLC.
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import Sailfish.Silica.private 1.0 as Private
import "private"

Dialog {
    id: page

    property string selectedPath
    property string title
    property var _homePage: page

    property bool checkLengthPath: false
    property var destination
    property int destinationAction: PageStackAction.Pop

    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias directorySort: fileModel.directorySort
    property alias caseSensitivity: fileModel.caseSensitivity

    property alias path: fileModel.path

    property bool _isHome: fileModel.path == StandardPaths.home
    property var _acceptDestinationTransit
    property string _folderText: path.split("/").pop()
    // 4096 - max path length 255 max filename length
    readonly property int _maxPathLength: 4096 - 255
    readonly property bool _tooLongPath: checkLengthPath && byteLength(_pressAndHoldPath ? _pressAndHoldPath : page.path) > _maxPathLength

    /*
     * When the user selects "back", the path needs to be changed to the previous one.
     * To do this, initialize selectedPath only if accepted.
     * But we also need to know if the path is selected by a long press
    */
    property string _pressAndHoldPath

    function byteLength(s) {
        return encodeURI(s).split(/%..|./).length - 1;
    }

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: __silica_applicationwindow_instance.contentItem
        targetPage: page
    }

    onStatusChanged: {
        if (page.status === PageStatus.Active) {
            if (!checkLengthPath && !acceptDestination && (page != _homePage)) {
                acceptDestination = pageStack.previousPage(page)
            }
        }
    }

    onAccepted: {
        if (_tooLongPath) {
            return
        }

        if (_pressAndHoldPath) {
            _homePage.selectedPath = _pressAndHoldPath
        } else if (!_homePage.selectedPath) {
            _homePage.selectedPath = page.path
        }
    }

    acceptDestinationAction: PageStackAction.Pop

    FileGridView {
        id: gridView

        anchors.fill: parent
        cellWidthForSmallScreen: parent.width
        currentIndex: -1

        header: DialogHeader {
            id: header

            spacing: 0

            //% "Back"
            cancelText: qsTrId("components_pickers-he-multiselect_dialog_back")

            width: parent.width
            // The empty string makes the button inactive. However, we need to add custom accept text
            acceptText: " "

            _children: Column {
                anchors.verticalCenter: parent.verticalCenter
                visible: fileModel.path !== StandardPaths.home

                Label {
                    text: page.title
                    color: highlighted
                           ? header.palette.highlightColor
                           : header.palette.primaryColor
                    width: header.width/2 - Theme.paddingLarge - Theme.horizontalPageMargin
                    truncationMode: TruncationMode.Fade
                    font.pixelSize: isPortrait ? Theme.fontSizeLarge : Theme.fontSizeMedium
                    horizontalAlignment: Qt.AlignRight
                }

                Label {
                    text: page._folderText
                    color: Theme.secondaryColor
                    width: header.width/2 - Theme.paddingLarge - Theme.horizontalPageMargin
                    truncationMode: TruncationMode.Fade
                    font.pixelSize: isPortrait ? Theme.fontSizeSmall : Theme.fontSizeExtraSmall
                    horizontalAlignment: Qt.AlignRight
                }
            }
        }

        model: FileModel {
            id: fileModel

            path: StandardPaths.home
            active: page.status === PageStatus.Active
            includeDirectories: true
            includeFiles: false
            sortBy: FileModel.SortByName
            directorySort: FileModel.SortDirectoriesBeforeFiles
        }

        delegate: GridItem {
            id: fileDelegate

            function onClickReaction() {
                gridView.currentIndex = -1

                var obj = pageStack.animatorPush('FolderPickerDialog.qml', {
                                                     path: model.absolutePath,
                                                     sortBy: fileModel.sortBy,
                                                     sortOrder: fileModel.sortOrder,
                                                     directorySort: fileModel.directorySort,
                                                     caseSensitivity: fileModel.caseSensitivity,
                                                     destination: page.destination,
                                                     destinationAction: page.destinationAction,
                                                     acceptDestination: _isHome ? page._acceptDestinationTransit : page.acceptDestination,
                                                     acceptDestinationAction: page.acceptDestinationAction,
                                                     checkLengthPath: checkLengthPath,
                                                     _homePage: _homePage,
                                                     _folderText: model.fileName,
                                                     title: page.title,
                                                     allowedOrientations: page.allowedOrientations
                                                 })
                obj.pageCompleted.connect(function(nextPage) {
                    if (page._pressAndHoldPath !== "") {
                        page._pressAndHoldPath = ""
                        page._folderText = page.path.split("/").pop()
                    }
                    nextPage.accepted.connect(function() {
                        page._dialogDone(DialogResult.Accepted)
                    })
                })
            }

            function onPressAndHoldReaction() {
                gridView.currentIndex = model.index
                page._folderText = model.fileName
                _pressAndHoldPath = model.absolutePath
            }

            onClicked: onClickReaction()
            onPressAndHold: onPressAndHoldReaction()

            highlighted: down || fileItem.selected
            contentWidth: fileItem.width
            contentHeight: fileItem.height
            propagateComposedEvents: true

            FileGridItem {
                id: fileItem

                onClicked: fileDelegate.onClickReaction()
                onPressAndHold: fileDelegate.onPressAndHoldReaction()

                width: gridView.cellWidth
                chunkWidth: gridView.chunkWidth
                fileName: model.absolutePath
                mimeType: model.mimeType
                size: model.size
                isDir: model.isDir
                created: model.created
                modified: model.modified
                selected: gridView.currentIndex === model.index
            }
        }

        ViewPlaceholder {
            //% "No subfolders"
            text: qsTrId("components_pickers-ph-no_subfolders")

            enabled: gridView.count === 0 && fileModel.populated
        }

        VerticalScrollDecorator {}
    }

    Component {
        id: tooLongPathErrorPageComponent

        Page {
            allowedOrientations: page.allowedOrientations
            canNavigateForward: false

            PageHeader {
                id: header

                //% "Error"
                title: qsTrId("components_pickers-la-error")
            }

            Column {
                anchors {
                    top: header.bottom
                    left: parent.left
                    margins: Theme.paddingLarge
                }
                width: parent.width - 2 * Theme.paddingLarge
                spacing: Theme.paddingMedium

                Label {
                    width: parent.width
                    color: Theme.highlightColor
                    font.pixelSize: Theme.fontSizeLarge
                    font.family: Theme.fontFamilyHeading
                    //% "Selected path is too long"
                    text: qsTrId("components_pickers-la-too_long_path_header")
                    wrapMode: Text.Wrap
                }

                Label {
                    width: parent.width
                    color: Theme.secondaryHighlightColor
                    font.pixelSize: Theme.fontSizeSmall
                    //% "Choose a shorter path"
                    text: qsTrId("components_pickers-la-too_long_path_body")
                    wrapMode: Text.Wrap
                }
            }
        }
    }

    Component.onCompleted: {
        if (_isHome) {
            _acceptDestinationTransit = acceptDestination
            acceptDestination = null
            forwardNavigation = false
        } else {
            if (checkLengthPath) {
                acceptDestination = Qt.binding(function () {
                    return _tooLongPath ? tooLongPathErrorPageComponent : destination
                })
                acceptDestinationAction = Qt.binding(function () {
                    return _tooLongPath ? PageStackAction.Animated : destinationAction
                })
            }
        }
    }

}
