/**
 * SPDX-FileCopyrightText: Copyright 2021-2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Nemo.FileManager 1.0
import Sailfish.FileManager 1.0
import Aurora.Controls 1.0

SilicaGridView {
    id: root

    property string initialPath
    property string description
    property bool gridLayout
    property bool selectionMode
    property bool isSelectOneItem
    property bool virtualFolder
    property bool searchPage
    property bool pickerPage
    property var extractor
    property bool extractMode: extractor ? extractor.extractMode : false
    property bool archiveMode
    property bool pasteMode: FileEngine.clipboardCount > 0
    property bool standardPathSelected
    property var standardPaths: []

    function clearSelectedFiles() {
        if (!root.selectionMode) {
            if (model.clearSelectedFiles != null) {
                model.clearSelectedFiles()
            }
        }

        FileManager.homePageSelectOneItem = false
        root.isSelectOneItem = false
    }

    function openThirdColumn(url, properties) {
        if (extractor != null) {
            extractor.openThirdColumn(url, properties)
        } else {
            return pageStack.animatorPush(url, properties)
        }
    }

    interactive: !isSelectOneItem
    bottomMargin: root.selectionMode ? toolBar.height : 0

    NumberAnimation {
        id: contentYAnimation

        target: root
        property: "contentY"
        from: root.contentY
        duration: 200
    }

    FileGridFooter {
        id: toolBar

        readonly property bool needMove: root.currentItem
                                         ? root.currentItem.y >= root.contentY + root.height - toolBar.height * 2
                                         : false

        readonly property bool needShow: root.selectionMode ||
                                         root.pasteMode ||
                                         root.extractMode ||
                                         root.isSelectOneItem

        property bool _needMoveRestore: needMove

        anchors.bottomMargin: needShow ? SafeZoneRect.insets.bottom : -toolBar.height

        _bottomPosition: true
        parentGridView: root
        buttonWidth: parent.width / 4
        standardPath: standardPaths.length > 0 && (standardPathSelected || StandardPaths.home === model.path)

        Behavior on anchors.bottomMargin {
            NumberAnimation {
                duration: contentYAnimation.duration
            }
        }

        onNeedShowChanged: {
            if (!root.currentItem || (!toolBar.needMove && !toolBar._needMoveRestore)) {
                return
            }

            if (toolBar.needShow) {
                contentYAnimation.to = root.contentY + toolBar.height
                contentYAnimation.start()
                toolBar._needMoveRestore = true
            } else {
                contentYAnimation.to = root.contentY - toolBar.height
                contentYAnimation.start()
                toolBar._needMoveRestore = false
            }
        }

        onPasteClicked: {
            FileEngine.pasteFiles(model.path, true, indexIfExists)
            clearSelectedFiles()
        }

        onCancelClicked: {
            if (pasteMode) {
                FileEngine.clearClipboard()
                clearSelectedFiles()
            } else if (extractMode) {
                extractor.archiveFile = ""
                extractor.extractMode = false
                clearSelectedFiles()
            }
        }

        onExtractClicked: {
            extractor.extractAllFiles(model.path)
            extractor.archiveFile = ""
            extractor.extractMode = false
            clearSelectedFiles()
        }

        onShareClicked: {
            shareAction.resources = model.selectedFilesToShare()
            shareAction.trigger()
        }

        onCopyClicked: {
            FileEngine.copyFiles(model.selectedFiles())
            root.isSelectOneItem = false
            FileManager.homePageSelectOneItem = false
        }

        onOpenMenuClicked: {
            info.file = model.selectedFiles()[0]
            openPopupMenu()
        }

        onDeleteClicked: {
            Remorse.popupAction(pageStack.currentPage, qsTrId("filemanager-la-deleting_all_selected"), function() {
                FileEngine.deleteFiles(model.selectedFiles(), true)
                clearSelectedFiles()
            })
        }
    }

    ShareAction {
        id: shareAction
    }

    // Width of image background in FileItem
    readonly property real cellBackgroundSize: Theme.iconSizeLauncher //Theme.iconSizeMedium + Theme.paddingLarge * 2

    readonly property real contentCellWidthForLargeScreen: cellBackgroundSize + Theme.paddingLarge * 3 // Column width in FileItem
                                                           + Theme.paddingSmall * 3 // horizontal padding of Row in FileItem

    readonly property real chunkWidth: {
        var columns = Math.floor(width / contentCellWidthForLargeScreen)
        return (width - columns * contentCellWidthForLargeScreen) / columns
    }
    readonly property real cellWidthForLargeScreen: contentCellWidthForLargeScreen + chunkWidth

    readonly property real cellHeightForLargeScreen: cellBackgroundSize
                                                     + Theme.paddingSmall * 9
                                                     + nameMetrics.height * 2 + sizeMetrics.height
    readonly property real cellHeightForSmallScreen: cellBackgroundSize + Theme.paddingSmall * 3
    property real cellWidthForSmallScreen

    cellHeight: gridLayout ? cellHeightForLargeScreen : cellHeightForSmallScreen
    cellWidth: gridLayout ? cellWidthForLargeScreen : cellWidthForSmallScreen

    currentIndex: -1
    displaced: Transition { NumberAnimation { properties: "x,y"; easing.type: Easing.InOutQuad; duration: 200 } }
    cacheBuffer: cellHeight * 16
    width: parent.width

    function openPopupMenu(selectedFile) {
        popupMenu.open(toolBar.itemAnchorPopup)
    }

    function closePopupMenu() {
        popupMenu.close()
    }

    Connections {
        target: pageStack.currentPage

        onStatusChanged: {
            if (status === PageStatus.Activating) {
                if (!FileManager.homePageSelectionMode && !FileManager.homePageSelectOneItem) {
                    root.clearSelectedFiles()
                    FileManager.homePageSelectionMode = false
                }
            }
        }
    }

    Connections {
        target: FileManager
        onSelectedFilesCleared: {
            if (model.clearSelectedFiles != null)
                model.clearSelectedFiles()

            FileManager.homePageSelectionMode = false
            FileManager.homePageSelectOneItem = false
            root.isSelectOneItem = false
        }
    }

    Connections {
        target: model
        onSelectedCountChanged: {
            if ((root.selectionMode || root.isSelectOneItem) && model.selectedCount !== 0) {
                var files = model.selectedFiles()
                for (var i = 0; i < files.length; i++) {
                    if (standardPaths.indexOf(files[i]) !== -1) {
                        root.standardPathSelected = true
                        return
                    }
                }
                root.standardPathSelected = false
            }
        }
    }

    onSelectionModeChanged: if (selectionMode) isSelectOneItem = false

    Rectangle {
        id: backgroundOverlay

        anchors.fill: parent
        visible: root.isSelectOneItem
        color: Theme.rgba("black",  Theme.opacityFaint)
        z: parent.z + 1

        MouseArea {
            anchors.fill: parent
            enabled: root.isSelectOneItem

            onClicked: FileManager.selectedFilesCleared()
            onPositionChanged: FileManager.selectedFilesCleared()
        }
    }

    FileInfo {
        id: info
    }

    PopupMenuMain {
        id: popupMenu

        onDetailsClicked: {
            openThirdColumn(Qt.resolvedUrl("DetailsPage.qml"), {
                                fileName: info.fileName,
                                mimeType: info.mimeType,
                                isDir: info.isDir,
                                modified: info.modified,
                                size: info.size,
                                path: info.absolutePath
                            })
        }

        onRenameClicked: {
            openThirdColumn(Qt.resolvedUrl("RenameDialog.qml"), {
                                oldPath: info.absolutePath
                            })
        }

        onCutClicked: {
            FileEngine.cutFiles(model.selectedFiles())
            root.isSelectOneItem = false
        }

        onExtractClicked: {
            extractor.archiveFile = info.absolutePath
            extractor.destinationPageName = "__archive_page"
            extractor.rejectedPageName = "__directory_page"
            if (needSelection)
                extractor.extractMode = true

            if (!needSelection) {
                extractor.extractAllFiles(model.path)
                extractor.archiveFile = ""
                clearSelectedFiles()
            }

            root.isSelectOneItem = false
        }

        onCompressClicked: {
            openThirdColumn(Qt.resolvedUrl("CreateArchiveDialog.qml"), {
                                extractorView: extractor,
                                files: model.selectedFiles(),
                                currentDirectoryPath: typeof model.path !== "undefined" && model.path.length > 0 ? model.path : ""
                            })
            clearSelectedFiles()
        }

        onLocationClicked: {
            if (!FileEngine.isSymLink(info.absolutePath)) {
                openDirectory(info.directoryPath)
            } else {
                if (openSymLink) {
                    openDirectory(info.directoryPath)
                } else {
                    //% "Following a symbolic link is prohibited"
                    errorNotification.show(qsTrId("filemanager-la-open_symlink_failed"))
                }
            }
            clearSelectedFiles()
        }

        onSelectModeClicked: {
            root.isSelectOneItem = false
        }

        virtualFolder: root.virtualFolder
        searchPage: root.searchPage
        standardPath: root.standardPaths.length > 0 && root.standardPathSelected
        isSelectOneItem: root.isSelectOneItem
        hasFiles: model.count > 0
        isUserDir: FileEngine.isUserDir(model.path)
        singleItemSelected: model.selectedCount === 1
        isArchive: info.isArchive
    }

    Behavior on contentY {
        enabled: !gridLayout
        NumberAnimation { duration: 20 }
    }

    FontMetrics {
        id: nameMetrics
        // This value should be taken from nameLabel in FileItem when the screen is large
        font.pixelSize: Theme.fontSizeExtraSmall
    }
    FontMetrics {
        id: sizeMetrics
        // This value should be taken from sizeLabel in FileItem when the screen is large
        font.pixelSize: Theme.fontSizeTiny
    }
}
