/****************************************************************************
**
** Copyright (c) 2022 - 2023 Open Mobile Platform LLC.
**
****************************************************************************/

import QtQuick 2.0
import QtDocGallery 5.0
import Sailfish.Silica 1.0

FileGridView {
    id: gridView

    property bool searchActive
    property var dialog
    property string headerPlaceholderText
    property string emptyPlaceholderText
    property var contentType
    property var itemModel
    property int selectedCount: 0
    property bool clearOnBackstep: true
    property bool glassOnly
    property int animationDuration: 0
    property string title
    property bool isDialog: true
    property int selectedFilesCounter: 0

    anchors.fill: parent
    cellWidthForSmallScreen: parent.width
    currentIndex: -1
    selectionEnabled: true

    header: Item {
        width: gridView.width
        height: isDialog ? dialogHeader.height : pageHeader.height

        SearchDialogHeader {
            id: dialogHeader

            onActiveFocusChanged: if (activeFocus) gridView.currentIndex = -1
            onActiveChanged: gridView.searchActive = active

            width: gridView.width
            dialog: gridView.dialog
            placeholderText: gridView.headerPlaceholderText
            model: itemModel
            contentType: gridView.contentType
            visible: (active || itemModel.count > 0) && isDialog
            selectedCount: gridView.selectedCount
            showBack: true
            _glassOnly: gridView.dialog._background
        }

        SearchPageHeader {
            id: pageHeader

            onActiveFocusChanged: if (activeFocus) gridView.currentIndex = -1
            onActiveChanged: gridView.searchActive = active

            width: gridView.width
            title: gridView.title
            placeholderText: gridView.headerPlaceholderText
            model: itemModel
            visible: !isDialog && (active || itemModel.count > 0)
        }
    }

    delegate: GridItem {
        id: fileDelegate

        onClicked: {
            if (isDialog) {
                fileItem.selected ? selectedFilesCounter-- : selectedFilesCounter++
                itemModel.updateSelected(index, !fileItem.selected)
            } else {
                _customSelectionHandler(itemModel, index, true)
            }
        }
        GridView.onAdd: AddAnimation { target: fileDelegate; duration: animationDuration }
        GridView.onRemove: RemoveAnimation { target: fileDelegate; duration: animationDuration }

        contentWidth: fileItem.width
        contentHeight: fileItem.height
        _backgroundColor: "transparent"
        propagateComposedEvents: true
        menu: menuComponent

        FileGridItem {
            id: fileItem

            onClicked: selectionEnabled
                       ? itemModel.updateSelected(index, !selected)
                       : _customSelectionHandler(itemModel, index, true)

            width: gridView.cellWidth
            chunkWidth: gridView.chunkWidth
            fileName: model.filePath
            size: model.fileSize
            modified: model.lastModified
            selectionEnabled: gridView.selectionEnabled
            selected: model.selected || _selectedModel.selected(model.filePath)
        }

        Connections {
            target: gridView
            onWidthChanged: if (gridView.largeScreen && _menuItem && _menuItem.parent) _calculateMenuOffset()
        }

        Component {
            id: menuComponent

            ContextMenu {
                MenuItem {
                    onClicked: Qt.openUrlExternally(model.filePath)
                    text: qsTrId("components_pickers-he-file_preview")
                }
            }
        }
    }

    ViewPlaceholder {
        text: emptyPlaceholderText
        enabled: !gridView.searchActive
                 && itemModel.count === 0
                 && (itemModel.status === DocumentGalleryModel.Finished
                     || itemModel.status === DocumentGalleryModel.Idle)
    }

    PullDownMenu {
        enabled: selectionEnabled
        visible: enabled
        MenuItem {
            onClicked: {
                for (var i = 0; i < itemModel.model.count; i++) {
                    if (!itemModel.model.get(i).selected) {
                        itemModel.updateSelected(i, true)
                        selectedFilesCounter++
                    }
                }
            }

            //% "Select all"
            text: qsTrId("pickers-me-select_all")
            visible: selectedFilesCounter >= 0 && selectedFilesCounter < itemModel.count && selectionEnabled
        }

        MenuItem {
            onClicked:  {
                for (var i = 0; i < itemModel.model.count; i++) {
                    if (itemModel.model.get(i).selected)
                        itemModel.updateSelected(i, false)
                }

                selectedFilesCounter = 0
            }

            //% "Clear selection"
            text: qsTrId("pickers-me-clear_selection")
            visible: selectedFilesCounter > 0 && selectionEnabled
        }
    }

    VerticalScrollDecorator {}
}
