/****************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2019-2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
** Copyright (c) 2018 – 2019 Jolla Ltd.
**
** License: Proprietary
**
****************************************************************************/
import QtQuick 2.6
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import Nemo.Thumbnailer 1.0

Row {
    id: root

    property real chunkWidth: 0
    property bool gridLayout
    readonly property var internalStorageRegExp: new RegExp("^" + StandardPaths.home)
    readonly property bool isInternalStorage: internalStorageRegExp.test(model.fileLocation)
    property string fileName: model.fileName
    property bool trackerMode
    property var mimeTypeContainer: !trackerMode ? [] : mimeTypeCallback()
    property string mimeType: !trackerMode ? model.mimeType : mimeTypeContainer["mimeType"]
    property double size: model.size
    property bool isDir: model.isDir
    property var created: model.created
    property var modified: model.modified
    property int nestedFiles: typeof model.nestedFiles !== "undefined" ? model.nestedFiles : 0
    property bool compressed
    property bool isArchive: !trackerMode ? FileEngine.isArchive(model.absolutePath) : mimeTypeContainer["isArchive"]
    property string searchText
    property var mimeTypeCallback

    readonly property bool isPicture: mimeType.indexOf("image") === 0 && supportedMimeTypesOfImages.indexOf(mimeType) !== -1
    readonly property bool isVideo: mimeType.indexOf("video") === 0

    property bool selectionEnabled
    property bool selected
    property bool showStorageIcon

    property bool displaySwitcherForDirectories: true

    // Current list of mimeTypes: https://confluence.omprussia.ru/x/2zCbAw
    readonly property var supportedMimeTypesOfImages: [
        "image/bmp",
        "image/gif",
        "image/ico",
        "image/icon",
        "image/jpeg",
        "image/pjpeg",
        "image/png",
        "image/svg+xml",
        "image/tiff",
        "image/vnd.microsoft.icon",
        "image/webp",
        "image/x-MS-bmp",
        "image/x-bmp",
        "image/x-ico",
        "image/x-icon",
    ]

    signal clicked
    signal pressAndHold

    padding: gridLayout ? Theme.paddingLarge : 0
    leftPadding: gridLayout ? padding + chunkWidth/2 : 0
    rightPadding: gridLayout ? padding + chunkWidth/2 : 0

    TextSwitch {
        id: switcher
        objectName: "TextSwitch"

        anchors.verticalCenter: gridLayout ? parent.top : parent.verticalCenter
        leftMargin: gridLayout ? Theme.paddingSmall : Theme.fontSizeExtraSmall
        width: gridLayout ? Theme.itemSizeExtraSmall / 2 : Theme.itemSizeExtraSmall
        height: gridLayout ? Theme.itemSizeExtraSmall : Theme.itemSizeSmall
        visible: selectionEnabled
                 ? displaySwitcherForDirectories ? true : !isDir
                 : false

        checked: root.selected
        automaticCheck: false

        onClicked: root.clicked()
        onPressAndHold: root.pressAndHold()
    }

    Item {
        id: emptyItem
        objectName: "EmptyItem"

        anchors.verticalCenter: gridLayout ? parent.top : parent.verticalCenter
        width: gridLayout ? Theme.itemSizeExtraSmall / 2 : Theme.itemSizeExtraSmall
        height: gridLayout ? Theme.itemSizeExtraSmall : Theme.itemSizeSmall
        visible: selectionEnabled && !displaySwitcherForDirectories ? isDir : false
    }

    Grid {
        id: grid

        width: selectionEnabled ? root.width - (displaySwitcherForDirectories ? switcher.width : emptyItem.width) : root.width

        rows: gridLayout ? 2 : 1
        columns: gridLayout ? 1 : 2

        horizontalItemAlignment: gridLayout ? Qt.AlignHCenter : Qt.AlignLeft
        verticalItemAlignment: gridLayout ? Qt.AlignTop : Qt.AlignVCenter

        spacing: Theme.paddingMedium

        Rectangle {
            id: imageBackground

            width: Theme.iconSizeMedium + Theme.paddingLarge * 2
            height: width
            gradient: Gradient {
                // Abusing gradient for inactive mimeTypes
                GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.1) }
                GradientStop { position: 1.0; color: "transparent" }
            }

            Loader {
                id: loader

                anchors.centerIn: parent
                sourceComponent: isPicture || isVideo ? pictureComponent : iconComponent
            }

            Loader {
                anchors {
                    top: parent.top
                    right: parent.right
                    topMargin: Theme.paddingSmall / 2
                    rightMargin: Theme.paddingSmall / 2
                }

                sourceComponent: storageIconComponent
                active: gridLayout && showStorageIcon
            }

            Component {
                id: pictureComponent

                Item {
                    width: imageBackground.width
                    height: imageBackground.height

                    Thumbnail {
                        objectName: "PictireThumbnail"

                        anchors.fill: parent
                        mimeType: root.mimeType
                        priority: Thumbnail.NormalPriority
                        source: fileName
                        sourceSize.width: parent.width
                        sourceSize.height: parent.height
                    }

                    HighlightImage {
                        objectName: "PictureHighlightImage"

                        anchors.fill: parent
                        visible: root.isVideo
                        source: root.isDir
                                ? "image://theme/icon-m-file-folder"
                                : Theme.iconForMimeType(root.mimeType)
                    }
                }
            }

            Component {
                id: iconComponent

                HighlightImage {
                    objectName: "IconHighlightImage"

                    anchors.centerIn: parent
                    source: {
                        if (root.isDir)
                            return "image://theme/icon-m-file-folder"

                        if (root.isArchive)
                            return "image://theme/icon-m-file-archive-folder"

                        return Theme.iconForMimeType(root.mimeType)
                    }
                }
            }

            Image {
                objectName: "CompressedImage"

                anchors {
                    top: parent.top
                    right: parent.right
                }

                visible: compressed
                source: {
                    var iconSource = "image://theme/icon-s-file-compressed-displaced"
                    return iconSource + (highlighted ? "?" + Theme.highlightColor : "")
                }
            }
        }

        Row {
            width: gridLayout
                   ? imageBackground.width + Theme.paddingMedium * 2
                   : grid.width - grid.spacing - imageBackground.width
            height: column.height

            Column {
                id: column
                
                width: parent.width - storageIcon.width - storageIcon.leftMargin
                spacing: Theme.paddingSmall
                bottomPadding: gridLayout
                               ? nameLabel.lineCount > 1 ? 0 : nameMetrics.height
                               : 0
                
                Label {
                    id: nameLabel
                    objectName: "NameLabel"
                    
                    width: gridLayout ? parent.width : parent.width - Theme.horizontalPageMargin
                    text: {
                        var _fileName = root.fileName.indexOf("/") >= 0
                                ? root.fileName.substring(root.fileName.lastIndexOf("/") + 1, root.fileName.length)
                                : root.fileName
                        
                        return root.searchText.length > 0 ? Theme.highlightText(_fileName, root.searchText.replace("*", ""), Theme.highlightColor)
                                                          : _fileName
                    }
                    
                    elide: gridLayout ? Text.ElideNone : Text.ElideMiddle
                    wrapMode: gridLayout ? Text.WrapAnywhere : Text.NoWrap
                    maximumLineCount: gridLayout ? 2 : 1
                    horizontalAlignment: gridLayout ? Qt.AlignHCenter : Qt.AlignLeft
                    font.pixelSize: gridLayout ? Theme.fontSizeExtraSmall : Theme.fontSizeMedium
                    color: dateLabel.highlighted ? Theme.highlightColor : Theme.primaryColor
                }
                
                Label {
                    id: sizeLabel
                    objectName: "SizeLabel"

                    width: nameLabel.width
                    text: root.isDir
                            //% "%n object(s)"
                          ? qsTrId("filemanager-la-nested_objects", root.nestedFiles)
                          : Format.formatFileSize(root.size)

                    color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                    horizontalAlignment: gridLayout ? Qt.AlignHCenter : Qt.AlignRight
                    font.pixelSize: gridLayout ? Theme.fontSizeTiny : Theme.fontSizeExtraSmall

                    Label {
                        id: dateLabel
                        objectName: "DateLabel"

                        width: parent.width
                        text: Qt.formatDateTime(root.modified || root.created, "d MMMM yyyy hh:mm")
                        color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                        horizontalAlignment: Qt.AlignLeft
                        font.pixelSize: gridLayout ? Theme.fontSizeTiny : Theme.fontSizeExtraSmall
                        visible: !gridLayout
                    }
                }
            }

            Loader {
                id: storageIcon

                readonly property int leftMargin: active ? Theme.horizontalPageMargin : 0

                anchors.verticalCenter: column.verticalCenter
                sourceComponent: storageIconComponent
                active: !gridLayout && showStorageIcon
            }
        }

        Component {
            id: storageIconComponent

            StorageIcon {
                isInternal: root.isInternalStorage
            }
        }

        FontMetrics {
            id: nameMetrics
            font.pixelSize: dateLabel.font.pixelSize
        }
    }
}
