/**
 * SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import QtQuick.Layouts 1.0
import Sailfish.Silica 1.0
import Sailfish.Media 1.0

Item {
    id: root

    property bool isPlaying
    property var artistAndSongText: { "artist": "", "song": "", "albumArt": "" }
    property bool nextEnabled
    property bool previousEnabled
    property bool playEnabled
    property bool pauseEnabled
    property color textColor: Theme.primaryColor
    property real buttonSize: Theme.iconSizeMedium

    readonly property bool isTablet: Screen.sizeCategory >= Screen.Large
    readonly property real _tabletImplicitWidth: Screen.width * 4 / 6
    readonly property real _maximumButtonsWidth: Theme.dp(326)
    readonly property real _albumArtSize: Theme.dp(142)

    signal playPauseRequested()
    signal nextRequested()
    signal previousRequested()

    implicitWidth: isTablet ? _tabletImplicitWidth : Screen.width - 2 * Theme.paddingLarge
    implicitHeight: _albumArtSize + 2 * Theme.paddingLarge

    onArtistAndSongTextChanged: {
        if (artistAndSongFadeAnimation.running) {
            artistAndSongFadeAnimation.complete()
        }

        artistAndSongFadeAnimation.artist = artistAndSongText.artist
        artistAndSongFadeAnimation.song = artistAndSongText.song
        artistAndSongFadeAnimation.albumArt = artistAndSongText.albumArt
        artistAndSongFadeAnimation.running = true
    }

    SequentialAnimation {
        id: artistAndSongFadeAnimation

        property string artist
        property string song
        property url albumArt

        FadeAnimation { targets: [titleLabel, subtitleLabel]; to: 0.0 }

        ScriptAction {
            script: {
                titleLabel.text = artistAndSongFadeAnimation.song
                subtitleLabel.text = artistAndSongFadeAnimation.artist
                albumArt.source = artistAndSongFadeAnimation.albumArt
            }
        }

        FadeAnimation { targets: [titleLabel, subtitleLabel]; to: 1.0 }
    }

    Rectangle {
        anchors.fill: parent
        radius: Theme.paddingMedium
        color: Theme.colorScheme === Theme.LightOnDark ? Theme.darkPrimaryColor : Theme.lightPrimaryColor
        opacity: Theme.opacityFaint
    }

    RowLayout {
        id: mainRow

        anchors.fill: parent
        spacing: 0

        Item {
            Layout.fillHeight: true
            Layout.preferredWidth: Theme.paddingLarge
        }

        AlbumArt {
            id: albumArt

            sourceSizeMax: root._albumArtSize
            Layout.preferredWidth: root._albumArtSize
            Layout.preferredHeight: root._albumArtSize
            Layout.alignment: Qt.AlignVCenter
        }

        Item {
            Layout.fillHeight: true
            Layout.preferredWidth: Theme.paddingMedium
        }

        Item {
            Layout.fillHeight: true
            Layout.fillWidth: true

            Label {
                id: titleLabel

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                    topMargin: Theme.paddingLarge
                }

                textFormat: Text.PlainText
                horizontalAlignment: implicitWidth > width ? Text.AlignLeft : Text.AlignHCenter
                color: Theme.primaryColor
                maximumLineCount: 1
                truncationMode: TruncationMode.Fade
                font.pixelSize: Theme.fontSizeLarge
            }

            Label {
                id: subtitleLabel

                anchors {
                    top: titleLabel.bottom
                    left: parent.left
                    right: parent.right
                    topMargin: Theme.paddingSmall
                }

                textFormat: Text.PlainText
                horizontalAlignment: implicitWidth > width ? Text.AlignLeft : Text.AlignHCenter
                color: Theme.secondaryColor
                maximumLineCount: 1
                truncationMode: TruncationMode.Fade
                font.pixelSize: Theme.fontSizeSmall
            }

            Item {
                anchors {
                    bottom: parent.bottom
                    bottomMargin: Theme.paddingLarge
                    horizontalCenter: parent.horizontalCenter
                }

                width: 3 * root.buttonSize + 2 * Theme.dp(42)
                height: root.buttonSize

                IconButton {
                    id: prevBtn

                    anchors.left: parent.left
                    enabled: root.previousEnabled
                    opacity: enabled ? 1.0 : 0.0

                    icon {
                        source: "image://theme/icon-m-previous2"

                        sourceSize {
                            width: root.buttonSize
                            height: root.buttonSize
                        }
                    }

                    onClicked: root.previousRequested()
                }

                IconButton {
                    id: playBtn

                    anchors.horizontalCenter: parent.horizontalCenter
                    enabled: root.isPlaying ? root.pauseEnabled : root.playEnabled

                    icon {
                        source: root.isPlaying ? "image://theme/icon-m-pause2" : "image://theme/icon-m-play2"

                        sourceSize {
                            width: root.buttonSize
                            height: root.buttonSize
                        }
                    }

                    onClicked: root.playPauseRequested()
                }

                IconButton {
                    id: nextBtn

                    anchors.right: parent.right
                    enabled: root.nextEnabled
                    opacity: enabled ? 1.0 : 0.0

                    icon {
                        source: "image://theme/icon-m-next2"

                        sourceSize {
                            width: root.buttonSize
                            height: root.buttonSize
                        }
                    }

                    onClicked: root.nextRequested()
                }
            }
        }

        Item {
            Layout.fillHeight: true
            Layout.preferredWidth: Theme.paddingLarge
        }
    }
}
