/****************************************************************************
**
** Copyright (c) 2018 – 2019 Jolla Ltd.
** Copyright (c) 2019 – 2022 Open Mobile Platform LLC.
**
** License: Proprietary
**
****************************************************************************/
import QtQuick 2.6
import Sailfish.Silica 1.0
import Nemo.Thumbnailer 1.0

Row {
    id: root

    property real chunkWidth: 0
    readonly property bool largeScreen: Screen.sizeCategory > Screen.Medium
    property string fileName: model.fileName
    property string mimeType: model.mimeType
    property double size: model.size
    property bool isDir: model.isDir
    property var created: model.created
    property var modified: model.modified
    property bool compressed

    property bool isPicture: mimeType.indexOf("image") == 0
    property bool isVideo: mimeType.indexOf("video") == 0

    property bool selectionEnabled
    property bool selected

    property bool displaySwitcherForDirectories: true

    signal clicked
    signal pressAndHold

    padding: largeScreen ? Theme.paddingLarge : 0
    leftPadding: largeScreen ? padding + chunkWidth/2 : 0
    rightPadding: largeScreen ? padding + chunkWidth/2 : 0

    TextSwitch {
        id: switcher
        objectName: "TextSwitch"

        anchors.verticalCenter: largeScreen ? parent.top : parent.verticalCenter
        leftMargin: largeScreen ? Theme.paddingSmall : Theme.fontSizeExtraSmall
        width: largeScreen ? Theme.itemSizeExtraSmall / 2 : Theme.itemSizeExtraSmall
        height: largeScreen ? Theme.itemSizeExtraSmall : Theme.itemSizeSmall
        visible: selectionEnabled
                 ? displaySwitcherForDirectories ? true : !isDir
                 : false

        checked: root.selected
        automaticCheck: false

        onClicked: root.clicked()
        onPressAndHold: root.pressAndHold()
    }

    Item {
        id: emptyItem
        objectName: "EmptyItem"

        anchors.verticalCenter: largeScreen ? parent.top : parent.verticalCenter
        width: largeScreen ? Theme.itemSizeExtraSmall / 2 : Theme.itemSizeExtraSmall
        height: largeScreen ? Theme.itemSizeExtraSmall : Theme.itemSizeSmall
        visible: selectionEnabled && !displaySwitcherForDirectories ? isDir : false
    }

    Grid {
        id: grid

        width: selectionEnabled ? root.width - (displaySwitcherForDirectories ? switcher.width : emptyItem.width) : root.width

        rows: largeScreen ? 2 : 1
        columns: largeScreen ? 1 : 2

        horizontalItemAlignment: largeScreen ? Qt.AlignHCenter : Qt.AlignLeft
        verticalItemAlignment: largeScreen ? Qt.AlignTop : Qt.AlignVCenter

        spacing: Theme.paddingMedium

        Rectangle {
            id: imageBackground

            width: Theme.iconSizeMedium + Theme.paddingLarge * 2
            height: width
            gradient: Gradient {
                // Abusing gradient for inactive mimeTypes
                GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.1) }
                GradientStop { position: 1.0; color: "transparent" }
            }

            Loader {
                id: loader

                anchors.centerIn: parent
                sourceComponent: isPicture || isVideo ? pictureComponent : iconComponent
            }

            Component {
                id: pictureComponent

                Item {
                    width: imageBackground.width
                    height: imageBackground.height

                    Thumbnail {
                        objectName: "PictireThumbnail"

                        anchors.fill: parent
                        mimeType: root.mimeType
                        priority: Thumbnail.NormalPriority
                        source: fileName
                        sourceSize.width: parent.width
                        sourceSize.height: parent.height
                    }

                    HighlightImage {
                        objectName: "PictureHighlightImage"

                        anchors.fill: parent
                        visible: root.isVideo
                        source: root.isDir
                                ? "image://theme/icon-m-file-folder"
                                : Theme.iconForMimeType(root.mimeType)
                    }
                }
            }

            Component {
                id: iconComponent

                HighlightImage {
                    objectName: "IconHighlightImage"

                    anchors.centerIn: parent
                    source: root.isDir
                            ? "image://theme/icon-m-file-folder"
                            : Theme.iconForMimeType(root.mimeType)
                }
            }

            Image {
                objectName: "CompressedImage"

                anchors {
                    top: parent.top
                    right: parent.right
                }

                visible: compressed
                source: {
                    var iconSource = "image://theme/icon-m-file-compressed"
                    return iconSource + (highlighted ? "?" + Theme.highlightColor : "")
                }
            }
        }

        Column {
            id: column

            width: largeScreen
                   ? imageBackground.width + Theme.paddingMedium * 2
                   : grid.width - grid.spacing - imageBackground.width

            spacing: Theme.paddingSmall
            bottomPadding: largeScreen
                           ? nameLabel.lineCount > 1 ? 0 : nameMetrics.height
                           : 0

            Label {
                id: nameLabel
                objectName: "NameLabel"

                width: largeScreen ? parent.width : parent.width - Theme.horizontalPageMargin
                text: root.fileName.indexOf("/") >= 0
                      ? root.fileName.substring(root.fileName.lastIndexOf("/") + 1, root.fileName.length)
                      : root.fileName

                elide: largeScreen ? Text.ElideNone : Text.ElideMiddle
                wrapMode: largeScreen ? Text.WrapAnywhere : Text.NoWrap
                maximumLineCount: largeScreen ? 2 : 1
                horizontalAlignment: largeScreen ? Qt.AlignHCenter : Qt.AlignLeft
                font.pixelSize: largeScreen ? Theme.fontSizeExtraSmall : Theme.fontSizeMedium
            }

            Label {
                id: sizeLabel
                objectName: "SizeLabel"

                property string dateString: Qt.formatDate(root.modified || root.created, largeScreen ? "d MMM yyyy" : "d MMMM yyyy")

                width: parent.width
                text: {
                    if (largeScreen) {
                        return root.isDir ? dateString : Format.formatFileSize(root.size)
                    } else {
                        return root.isDir ? dateString
                                            //: Shows size and modification/created date, e.g. "15.5MB, 02/03/2016"
                                            //% "%1, %2"
                                          : qsTrId("filemanager-la-file_details").arg(Format.formatFileSize(root.size)).arg(dateString)
                    }
                }

                color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                horizontalAlignment: largeScreen ? Qt.AlignHCenter : Qt.AlignLeft
                font.pixelSize: largeScreen ? Theme.fontSizeTiny : Theme.fontSizeExtraSmall
            }
        }

        FontMetrics {
            id: nameMetrics
            font.pixelSize: sizeLabel.font.pixelSize
        }
    }
}
