/****************************************************************************************
**
** Copyright (c) 2022 Open Mobile Platform LLC.
** Copyright (C) 2013 Jolla Ltd.
** All rights reserved.
**
** This file is part of Sailfish Silica UI component package.
**
** You may use this file under the terms of BSD license as follows:
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are met:
**     * Redistributions of source code must retain the above copyright
**       notice, this list of conditions and the following disclaimer.
**     * Redistributions in binary form must reproduce the above copyright
**       notice, this list of conditions and the following disclaimer in the
**       documentation and/or other materials provided with the distribution.
**     * Neither the name of the Jolla Ltd nor the
**       names of its contributors may be used to endorse or promote products
**       derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
** DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
** ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as SilicaPrivate
import QtGraphicalEffects 1.0

import "private"
import "private/Util.js" as Util

PulleyMenuBase {
    id: pullDownMenu

    property real topMargin: Theme.itemSizeSmall
    property real bottomMargin: _menuLabel ? 0 : Theme.paddingLarge
    property real _contentEnd: contentColumn.height + bottomMargin
    property Item _menuLabel: {
        var lastChild = contentColumn.visible && Util.childAt(contentColumn, width/2, contentColumn.height-1)
        if (lastChild && lastChild.hasOwnProperty("__silica_menulabel")) {
            return lastChild
        }
        return null
    }
    property real _bottomDragMargin: (_menuLabel ? _menuLabel.height : 0) + bottomMargin
    default property alias _content: contentColumn.children
    property real _originalPos: flickable.contentY - _finalPosition

    property real topPadding: 0
    spacing: 0
    y: flickable.originY - height + topPadding

    _contentColumn: contentColumn
    _isPullDownMenu: true
    _inactiveHeight: 0
    _activeHeight: contentColumn.height + topMargin + bottomMargin
    _inactivePosition: Math.round(flickable.originY - (_inactiveHeight + spacing))
    _finalPosition: _inactivePosition - _activeHeight
    _menuIndicatorPosition: height - _menuItemHeight + Theme.paddingSmall - spacing - _originalPos
    _highlightIndicatorPosition: Math.min(height - Math.min(_dragDistance, _contentEnd) - spacing,
        _menuIndicatorPosition - (_dragDistance/(_menuItemHeight+_bottomDragMargin)*(Theme.paddingSmall+_bottomDragMargin))) + _originalPos
    width: (flickable.width ? Math.min(flickable.width, screen.sizeCategory > Screen.Medium ? Screen.width*0.7 : Screen.width) : Screen.width)
           - 2 * Theme.horizontalPageMargin
    highlightItemRadius: Theme.paddingSmall

    property Component background: Rectangle {
        id: bg

        width: parent.width
        height: _activeHeight
        opacity: pullDownMenu.active ? Theme.opacityLow : 0.0
        color: pullDownMenu.backgroundColor
    }

    function _resetPosition() {
        flickable.contentY = _inactivePosition
    }
    on_AtInitialPositionChanged: {
        if (!_atInitialPosition && !flickable.moving && _page && _page.orientationTransitionRunning) {
            // If this flickable has a context menu open, the menu visibility takes precedence over initial position reset
            if (('__silica_contextmenu_instance' in flickable) && flickable.__silica_contextmenu_instance && flickable.__silica_contextmenu_instance._open) {
                return
            }

            // If the menu was at the inactive position before the orientation transition, it should still be afterward
            SilicaPrivate.Util.asyncInvoke(_resetPosition)
        }
    }

    property Component menuIndicator // Remains for API compatibility
    onMenuIndicatorChanged: console.log("WARNING: PullDownMenu.menuIndicator is no longer supported.")

    property Item _pageStack: Util.findPageStack(pullDownMenu)

    onActiveChanged: {
        if (_pageStack) {
            _pageStack._activePullDownMenu = active ? pullDownMenu : null
        }
    }

    // pulleyIndicator
    Rectangle {
        y: flickable.contentY - flickable.originY > 0 ? _activeHeight : _originalPos
        width: parent.width
        opacity: 0.8 - Math.min(mask.height / _activeHeight, 0.8)
        height: Math.max(Theme.dp(8), flickable.contentY < flickable.originY ?
                             Math.abs(flickable.contentY - flickable.originY) : 0) // 8 by design
        color: pullDownMenu.highlightColor
        radius: Theme.paddingMedium
    }

    Item {
        id: mask

        y: _originalPos
        width: parent.width
        height: _dragDistance

        layer.enabled: true
        layer.effect: OpacityMask {
            maskSource: Rectangle {
                width: mask.width
                height: mask.height
                radius: Theme.paddingMedium
            }
        }

        Column {
            id: contentColumn

            property int __silica_pulleymenu_content

            property real menuContentY: pullDownMenu.active ? pullDownMenu.height - _dragDistance - pullDownMenu.spacing : -1
            onMenuContentYChanged: {
                if (menuContentY >= 0) {
                    if (flickable.dragging && !_bounceBackRunning) {
                        _highlightMenuItem(contentColumn, menuContentY - y - _originalPos + _menuItemHeight)
                    } else if (quickSelect){
                        _quickSelectMenuItem(contentColumn, menuContentY - y - _originalPos + _menuItemHeight)
                    }
                }
            }

            y: pullDownMenu.topMargin - _originalPos
            width: parent.width
            visible: active
        }
    }

    Binding {
        target: flickable
        property: "topMargin"
        value: active ? pullDownMenu.height : _inactiveHeight + spacing
    }

    // Create a bottomMargin to fill the remaining space in views
    // with content size < view height.  This allows the view to
    // be positioned above the bottomMargin even when
    // its content is smaller than the available space.
    Binding {
        when: !flickable.pushUpMenu  // If there is a PushUpMenu then it will take care of it.
        target: flickable
        property: "bottomMargin"
        value: Math.max(flickable.height - flickable.contentHeight - (_inactiveHeight + pullDownMenu.spacing), 0)
    }

    // If the content size is less than view height and there is no
    // push up menu, then we must also prevent moving in the wrong direction
    property real _maxDragPosition: Math.min(flickable.height - flickable.contentHeight, _inactivePosition)

    function _addToFlickable(flickableItem) {
        if (flickableItem.pullDownMenu !== undefined) {
            flickableItem.pullDownMenu = pullDownMenu
        } else {
            console.log('Warning: PullDownMenu must be added to an instance of SilicaFlickable.')
        }
    }

    // for testing
    function _menuContentY() {
        return contentColumn.menuContentY
    }

    Component.onCompleted: {
        if (background) {
            background.createObject(mask, {"z": -2})
            highlightItem.parent = mask
        }
        _updateFlickable()
    }
}
