/****************************************************************************
**
** Copyright (c) 2022 Open Mobile Platform LLC.
**
****************************************************************************/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Nemo.FileManager 1.0

Dialog {
    id: dialog

    property alias filename: textArea.text
    property string path
    property var invalidCharacters: ["/", "\0"]

    readonly property int _maxFileNameLength: 255
    readonly property bool _error: _hasInvalidCharacters || _tooLong || _empty || _hasTrailingSpace
    readonly property bool _hasInvalidCharacters: hasInvalidCharacters(textArea.text, invalidCharacters)
    readonly property bool _tooLong: byteLength(filename) > _maxFileNameLength
    readonly property bool _empty: textArea.length == 0
    readonly property bool _hasTrailingSpace: /(^\s+|\s+$)/.test(textArea.text)

    function byteLength(s) {
        return encodeURI(s).split(/%..|./).length - 1;
    }

    function hasInvalidCharacters(str, restrictedSymbols) {
        for (var i = 0; i < restrictedSymbols.length; i++) {
            if (str.indexOf(restrictedSymbols[i]) !== -1) {
                return true
            }
        }
        return false
    }

    canAccept: !_error

    DialogHeader {
        id: header

        //% "Back"
        cancelText: qsTrId("components_pickers-he-multiselect_dialog_back")
    }

    TextArea {
        id: textArea

        readonly property real marginsHeight: textTopMargin + _bottomMargin

        function _suffixForFileName(fileName) {
            var suffix = FileEngine.extensionForFileName(fileName)
            return suffix !== "" ?  "." + suffix : suffix
        }

        onFocusChanged: if (focus) select(0, text.length - _suffixForFileName(text).length)

        anchors.top: header.bottom
        height: {
            const lineCount = Math.min(5, textArea._editor.lineCount)
            const oneLineHeight = metrics.height + Theme.dp(1)

            // If keyboard is opened we need to calculate available space for textarea
            if (virtualKeyboardObserver.opened) {
                var heightForTextArea = dialog.height - header.height

                // We calculate maximal line count which could be displayed on the screen
                if ((textTopMargin + oneLineHeight * lineCount) > heightForTextArea) {
                    for (var i =  lineCount - 1; i > 1; i--) {
                        if ((textTopMargin + oneLineHeight * i) <= heightForTextArea) {
                            return marginsHeight + oneLineHeight * i
                        }
                    }
                    return marginsHeight + oneLineHeight
                }
            }
            return marginsHeight + oneLineHeight * Math.max(1, lineCount)
        }
        width: parent.width
        errorHighlight: dialog._error
        _errorIcon.visible: false
        //% "Enter filename"
        placeholderText: qsTrId("components_pickers-ph-enter_filemame")
        label: placeholderText
        description: {
            if (dialog._empty) {
                //% "Filename is empty"
                return qsTrId("components_pickers-la-error_empty")
            } else if (dialog._hasInvalidCharacters) {
                //% "Filename contains invalid characters"
                return qsTrId("components_pickers-la-error_invalid_characters")
            } else if (dialog._tooLong) {
                //% "Filename is too long"
                return qsTrId("components_pickers-la-error_too_long")
            } else if (dialog._hasTrailingSpace) {
                //% "Filename cannot start or end with spaces"
                return qsTrId("components_pickers-la-error_trailing_spaces")
            } else {
                return ""
            }
        }

        EnterKey.iconSource: "image://theme/icon-m-enter-close"
        EnterKey.onClicked: focus = false
    }

    FontMetrics {
        id: metrics

        font: textArea._editor.font
    }

    Private.VirtualKeyboardObserver {
        id: virtualKeyboardObserver

        active: true
        orientation: dialog.orientation
    }
}
