/****************************************************************************
**
** Copyright (C) 2021 Open Mobile Platform LLC
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import Sailfish.Share 1.0
import Sailfish.FileManager 1.0

Page {
    id: page
    readonly property int selectionCount: fileProxyModel ? fileProxyModel.selectedCount : 0

    property bool actionInProgress
    property alias selectionModel: fileProxyModel.sourceModel
    property alias filePathRole: fileProxyModel.filePathRole

    on_NavigationChanged: fileProxyModel.clearSelectedFiles()
    onStatusChanged: {
        if (status === PageStatus.Deactivating) {
            if (!actionInProgress) {
                fileProxyModel.clearSelectedFiles()
            }
        }
    }

    function _moveClicked() {
        actionInProgress = true
        FileEngine.cutFiles(fileProxyModel.selectedFiles)
        pageStack.pop()
    }

    function _copyClicked() {
        actionInProgress = true
        FileEngine.copyFiles(fileProxyModel.selectedFiles)
        pageStack.pop()
    }

    function _deleteClicked() {
        actionInProgress = true
        //% "Deleting all selected"
        Remorse.popupAction(page, qsTrId("filemanager-la-deleting_all_selected"), function() {
            FileEngine.deleteFiles(fileProxyModel.selectedFiles, true)
            pageStack.pop()
        })
    }

    function _shareClicked() {
        actionInProgress = true
        shareAction.resources = fileProxyModel.selectedFilesToShare()
        shareAction.trigger()
    }

    backNavigation: !FileEngine.busy

    FileProxyModel {
        id: fileProxyModel
    }

    FileGridView {
        id: gridView
        objectName: "MultiselectionFileGridView"

        anchors {
            top: parent.top
            bottom: dockedPanel.top
        }
        width: parent.width

        cellWidthForSmallScreen: parent.width

        header: Column {
            width: parent.width
            PageHeader {
                objectName: "SelectedItemsPageHeader"

                title: selectionCount ? //: Selected items
                                        //% "Selected %n"
                                        qsTrId("filemanager-he-select_item", selectionCount)
                                      : //: Item selection header, no currently selected items
                                        //% "Selected"
                                        qsTrId("filemanager-he-zero_selected_items")
            }
            // Padding to match design
            Item {
                width: parent.width
                height: Theme.paddingSmall * 5
                opacity: gridView.largeScreen ? 1 : 0

                Behavior on opacity { FadeAnimation {} }
            }
        }
        selectionEnabled: true

        clip: dockedPanel.expanded
        model: fileProxyModel

        delegate: GridItem {
            id: fileDelegate

            function _toggleSelection() {
                fileProxyModel.toggleSelectedFile(model.index)
            }

            contentWidth: fileItem.width
            contentHeight: fileItem.height

            _backgroundColor: "transparent"

            propagateComposedEvents: true

            onClicked: _toggleSelection()
            onPressAndHold: _toggleSelection()
            GridView.onRemove: if (page.status === PageStatus.Active) animateRemoval(fileDelegate)

            FileGridItem {
                id: fileItem
                objectName: "MultiselectionFileGridItem"

                width: gridView.cellWidth
                chunkWidth: gridView.chunkWidth

                fileName: model.absolutePath
                selectionEnabled: gridView.selectionEnabled
                selected: model.isSelected

                onPressAndHold: fileDelegate._toggleSelection()
                onClicked: {
                    model.isSelected = selected
                    fileDelegate._toggleSelection()
                }
            }
        }

        PullDownMenu {
            busy: FileEngine.busy

            MenuItem {
                objectName: "DeselectAllMenuItem"

                //: Deselect all items
                //% "Deselect all"
                text: qsTrId("filemanager-me-deselect_all_items")
                visible: selectionCount

                onClicked: fileProxyModel.clearSelectedFiles()
            }

            MenuItem {
                objectName: "SelectAllMenuItem"

                //: Select all items
                //% "Select all"
                text: qsTrId("filemanager-me-select_all_items")
                visible: fileProxyModel.count > 0 && selectionCount < fileProxyModel.count

                onClicked: fileProxyModel.selectAllFiles()
            }
        }

        VerticalScrollDecorator {}
    }

    DockedPanel {
        id: dockedPanel
        objectName: "MultiselectionDockedPanel"

        readonly property int buttonWidth: dockedPanel.width/4

        width: parent.width
        height: Theme.itemSizeLarge
        dock: Dock.Bottom
        open: selectionCount

        Image {
            objectName: "GradientImage"

            fillMode: Image.PreserveAspectFit
            source: "image://theme/graphic-gradient-edge"
            anchors.fill: parent
        }

        Row {
            DockedItem {
                objectName: "DeleteDockedItem"

                width: dockedPanel.buttonWidth
                icon.source: "image://theme/icon-m-delete"

                onClicked: _deleteClicked()
            }

            DockedItem {
                objectName: "ShareDockedItem"

                width: dockedPanel.buttonWidth
                enabled: fileProxyModel.selectedCountFolders === 0
                icon.source: "image://theme/icon-m-share"

                onClicked: _shareClicked()

                ShareAction {
                    id: shareAction
                }
            }

            DockedItem {
                objectName: "CutDockedItem"

                width: dockedPanel.buttonWidth
                icon.source: "image://theme/icon-m-shortcut"

                onClicked: _moveClicked()
            }

            DockedItem {
                objectName: "CopyDockedItem"

                width: dockedPanel.buttonWidth
                icon.source: "image://theme/icon-m-copy"

                onClicked: _copyClicked()
            }
        }
    }
}
